##########################################################
## DTSM functions

## JSZ functions - used in jsz_est and jsz_analysis

y.loads <- function(y.mats, rinfQ, lamQ, Omega, W, jsz.mats, rn.flag = FALSE, K0P.cP=NULL, K1P.cP=NULL, conv.flag = TRUE) {
  ## yield loadings
  ## y.mats - for which maturities? (in periods)
  ## jsz.mats - maturities associated with yields that are measured without error (W)

  jsz.loads <- jsz.loadings(W, diag(lamQ), rinfQ, Omega, jsz.mats)
  ##[BcP, AcP, AX, BX]  -> need AX and BX  

  parP <- jsz.rotation(W, diag(lamQ), rinfQ, dt=1, BX=jsz.loads$BX, AX=jsz.loads$AX)
  ##[K0Q.cP, K1Q.cP, rho0.cP, rho1.cP]
  ## -> need rho0 and rho1

  if (conv.flag) {
    ## with convexity
    H0 <- Omega
  } else {
    ## without convexity
    H0 <-  matrix(0,N,N)
  }

  if (rn.flag) {
    ## risk-neutral yields
    loads <- gaussian.loadings(y.mats, K0P.cP, K1P.cP, H0, parP$rho0.cP, parP$rho1.cP)
  } else {
    ## fitted yields
    loads <- gaussian.loadings(y.mats, parP$K0Q.cP, parP$K1Q.cP, H0, parP$rho0.cP, parP$rho1.cP)
  }
  y.loads <- loads
}

y2f.loads <- function(mats, rinfQ, lamQ, Omega, W, jsz.mats, rn.flag = FALSE, K0P.cP=NULL, K1P.cP=NULL, conv.flag = TRUE) {
  ## calculate forward rate loadings for consecutive maturities
  ## by first getting yield loadings and then transforming to forward rate loadings
  J <- length(mats)
  yloads <- y.loads(mats, rinfQ, lamQ, Omega, W, jsz.mats, rn.flag, K0P.cP, K1P.cP, conv.flag)
  Af <- yloads$A
  Af[2:J] <- (mats[2:J]*Af[2:J]-mats[1:(J-1)]*Af[1:(J-1)])/(mats[2:J]-mats[1:(J-1)])
  Bf <- yloads$B
  Bf[,2:J] <- (rep(1,N)%*%t(mats[2:J])*Bf[,2:J]-rep(1,N)%*%t(mats[1:(J-1)])*Bf[,1:(J-1)])/(rep(1,N)%*%t(mats[2:J]-mats[1:(J-1)]))
  y2f.loads <- list(Af=Af, Bf=Bf)
}

f.loads <- function(h.from, h.to, rinfQ, lamQ, Omega, W, jsz.mats, rn.flag = FALSE, K0P.cP=NULL, K1P.cP=NULL, conv.flag = TRUE) {
  ## forward rate loadings

  if (length(h.from)>1)
    stop("not yet implemented for vector h.from -- must be scalar")
  
  yl <- y.loads(y.mats=c(h.from,h.to), rinfQ, lamQ, Omega, W, jsz.mats, rn.flag, K0P.cP, K1P.cP, conv.flag)

  Af <- (h.to*yl$A[2]-h.from*yl$A[1])/(h.to-h.from)
  Bf <- (h.to*yl$B[,2]-h.from*yl$B[,1])/(h.to-h.from)  
  f.loads <- list(A=Af, B=Bf)
}


f.jsz <- function(rinfQ, lamQ, Omega, W, P, h.from, h.to, jsz.mats, rn.flag = FALSE, K0P.cP=NULL, K1P.cP=NULL, conv.flag = TRUE) {
  ## forward rates  (actual or risk-neutral)
  ## input: parameters -- rinfQ,lamQ,Omega,physical dynamics
  ##        W -- which linear combinations are priced without error?
  ##        P -- time series of pricing factors
  ##        h.from, h.to -- beginning and end of forward maturity period
  ##        mats -- maturities in the data
  ##        rn.flag -- calculate actual (FALSE-default) or
  ##                   risk-neutral (TRUE) forward rate?
  ##        conv.flag -- include convexity term (TRUE-default) or
  ##                     calc yields as average short rates (FALSE)
  
  if (length(h.from)>1)
    stop("not yet implemented for vector h.from -- must be scalar")

  T <- nrow(P)

  ## obtain forward rate loadings
  loads <- f.loads(h.from, h.to, rinfQ, lamQ, Omega, W, jsz.mats, rn.flag, K0P.cP, K1P.cP, conv.flag)
  f.jsz <- rep(1,T) * loads$A + P %*% loads$B
  
  if (exists("start.date")) 
    f.jsz <- ts(f.jsz, start=start.date, frequency=freq)
  f.jsz
}

jsz.llk <- function (yields.o, W, lamQ.X, rinfQ, K0P.cP, K1P.cP, Sigma.cP, mats, dt=1, sigma.e=NA) {
# Compute the likelihood for a Gaussian term structure.
# Source "A New Perspective on Gaussian Dynamic Term Structure Models" by Joslin, Singleton and Zhu
#
# INPUTS:
# yields.o   : (T+1)*J,  matrix of observed yields (first row are t=0 observations, which likelihood conditions on)
# mats       : 1*J,      maturities in years
# dt         : scalar,   length of period in years
#
# W          : N*J,      vector of portfolio weights to fit without error.
# K1Q_X      : N*N,      normalized latent-model matrix (does not have to be diagonal, see form below)
# rinfQ      : scalar,   long-run mean under Q, of the annualized short rate
# K0P.cP     : N*1,      
# K1P.cP     : N*N,      
# Sigma.cP   : N*N,      positive definite matrix that is the covariance of innovations to cP
#
# Compute likelihood conditioned on first observation!
# output
# 
# llk        : T*1       time series of -log likelihoods (includes 2-pi constants)
# AcP        : 1*J       yt = AcP' + BcP'*Xt  (yt is J*1 vector)
# BcP        : N*J       AcP, BcP satisfy internal consistency condition that AcP*W' = 0, BcP*W' = I_N
# AX         : 1*J       yt = AX' + BX'*Xt  
# BX         : N*J       Xt is the 'jordan-normalized' latent state
#
#
# The model takes the form:
#   r(t) = rho0.cP + rho1.cP'*cPt
#        = rinfQ + 1'*Xt  (Xt is the 'jordan-normalized' state
#        = 1 period discount rate (annualized)
#
# Under Q:
#   X(t+1) - X(t)   =          K1Q.X*X(t)  + eps_X(t+1),   cov(eps_X(t+1)) = Sigma_X
#   cP(t+1) - cP(t) = K0Q.cP + K1Q.cP*X(t) + eps_cP(t+1),  cov(eps_cP(t+1)) = Sigma.cP
#   where Sigma_X is chosen to match Sigma.cP 
#
# Under P:
#   cP(t+1) - cP(t) = K0P.cP + K1P.cP*X(t) + eps_cP(t+1),  cov(eps_cP(t+1)) = Sigma.cP
#
# Model yields are given by:
#   yt^m = AcP' + BcP'*cPt  (J*1)
# And observed yields are given by:
#  yt^o = yt^m + epsilon.e(t)
# where V*epsilon.e~N(0,sigma.e^2 I_(J-N))
# and V is an (J-N)*J matrix which projects onto the span orthogonal to the
# row span of W.  This means errors are orthogonal to cPt and cPt^o = cPt^m.
#
    
    
########################################################################
# Setup 
T <- nrow(yields.o)-1
J <- ncol(yields.o)
N <- nrow(W)
cP <- yields.o %*% t(W) # (T+1)*N, cP stands for math caligraphic P.
########################################################################

########################################################################
# COMPUTE THE Q-LIKELIHOOD:
# First find the loadings for the model:
# yt = AcP' + BcP'*cPt, AcP is 1*J, BcP is N*J
loads <- jsz.loadings(W, diag(lamQ.X), rinfQ, Sigma.cP, mats, dt)
##[BcP, AcP, AX, BX]
parP <- jsz.rotation(W, diag(lamQ.X), rinfQ, dt, loads$BX, loads$AX)
##[K0Q.cP, K1Q.cP, rho0.cP, rho1.cP]

#test.loads <- gaussian.loadings(mats, parP$K0Q.cP, parP$K1Q.cP, Sigma.cP, parP$rho0.cP, parP$rho1.cP, dt)
## these should be the same as loads
## correct loadings: satisfy  P = W*A + W*B*P => W*A = 0, W*B = I
#test.loads$A %*% t(W)   ## this seems right
#test.loads$B %*% t(W)   ## this seems right
#loads$AcP %*% t(W)      ## not quite zero
#loads$BcP %*% t(W)      ## this seems right

yields.m =  rep(1,T+1)%*%loads$AcP + cP %*% loads$BcP # (T+1)*J, model-implied yields
yield.errors = yields.o[2:(T+1),] - yields.m[2:(T+1),]; # T*J
square_orthogonal_yield.errors = yield.errors^2; # T*J, but N-dimensional projection onto W is always 0, so effectively (J-N) dimensional

# Compute optimal sigma.e if it is not supplied
if (is.na(sigma.e))
    sigma.e = sqrt( sum(square_orthogonal_yield.errors)/(T*(J-N)) )

llkQ = .5*rowSums(square_orthogonal_yield.errors)/sigma.e^2 + (J-N)*.5*log(2*pi) + .5*(J-N)*log(sigma.e^2) # 1*T
########################################################################


########################################################################
# COMPUTE THE P-LIKELIHOOD:
## if (concentrateK0PK1P) {
  ## Run OLS to obtain maximum likelihood estimates of K0P, K1P
##  res <- ar.ols(cP,order=1,aic=FALSE,demean=FALSE,intercept=TRUE)
##   K1P.cP <- res$ar[,,]
##  K0P.cP <- res$x.intercept
## }

innovations = t(cP[2:nrow(cP),]) - (K0P.cP%*%matrix(1,1,T) + K1P.cP%*%t(cP[1:(nrow(cP)-1),])) # N*T

if (rcond(Sigma.cP)<.Machine$double.eps) {
  cat("SINGULARITY -- matrix (Sigma.cP) is computationally singular\n")
  cat("rcond(Sigma.cP) = ", rcond(Sigma.cP),"\n")
  print("Sigma.cP = ")
  print(Sigma.cP)
  stop("singularity error in jsz.llk\n")
}

llkP = .5*N*log(2*pi) + .5*log(det(Sigma.cP)) + .5*colSums(innovations*solve(Sigma.cP, innovations)) # 1*T

########################################################################

  jsz.llk <- list(llk=t(llkQ + llkP), AcP=loads$AcP, BcP=loads$BcP, AX=loads$AX, BX=loads$BX, K0P.cP=K0P.cP, K1P.cP=K1P.cP, sigma.e=sigma.e, K0Q.cP=parP$K0Q.cP, K1Q.cP = parP$K1Q.cP, rho0.cP=parP$rho0.cP, rho1.cP=parP$rho1.cP, cP=cP, llkP=llkP, llkQ=llkQ)
                  
}

########################################################################
########################################################################

jsz.loadings <- function(W, K1Q.X, rinfQ, Sigma.cP, mats, dt=1) {
# Inputs:
#   mats       : 1*J,      maturities in years
#   dt         : scalar,   length of period in years
#   W          : N*J,      vector of portfolio weights to fit without error.
#   K1Q.X      : N*N
#   rinfQ      : scalar,   the long run mean under Q of the annualized short rate
#   Sigma.cP, Sigma_X : N*N  covariance of innovations. PROVIDE ONE OR THE OTHER
#
# Returns:
#   AcP : 1*J
#   BcP : N*J
#   AX  : 1*J
#   BX  : N*J
#   Sigma_X : N*N
#
# This function:
# 1. Compute the loadings for the normalized model:
#     X(t+1) - X(t) = K1Q.X*X(t) + eps_X(t+1), cov(eps_X)=Sigma_X
#     and r(t) = rinfQ + 1.X(t)  
#     where r(t) is the annualized short rate, (i.e. price of 1-period zero coupon bond at time t is exp(-r(t)*dt))
#    If Sigma_X is not provided, it is solved for so that Sigma.cP (below) is matched.
#    yt = AX' + BX'*Xt
#
# 2. For cPt = W*yt and the model above for Xt, find AcP, BcP so that
#    yt = AcP' + BcP'*cPt
#
#

J <- length(mats)
N <- nrow(K1Q.X)
K0Q.X <- matrix(0,N,1)
rho0d <- rinfQ
rho1d <- rep(1,N)
mats_periods <- round(mats/dt)
M = max(mats_periods)

############################################################
## If Sigma.cP is provided, we need to compute Sigma_X by 
## first computing BX
##
## First compute the loadings ignoring the convexity term -- BX will be correct
## yt = AX' + BX'*Xt  
## yt is J*1
## AX is 1*J
## BX is N*J
## Xt is N*1
## W is N*J
##
## cPt = W*yt  (cPt N*1, W is N*J) 
##     = W*AX' + W*BX'*Xt
##     = WAXp + WBXp*Xt
##
## Substituting:
## yt = AX' + BX'*(WBXp\(cPt - WAXp))
##    = (I - BX'*(WBXp\WAXp))*AX' + BX'*WBXp\cPt
##    = AcP' + BcP'*cPt
## where AcP = AX*(I - BX'*(WBXp\WAXp))'
##       BcP = (WBXp)'\BX
##
## Sigma.cP = W*BX'*Sigma_X*(W*BX')'
## Sigma_X = (W*BX')\Sigma.cP/(W*BX')'

loads.X.prelim <- gaussian.loadings(mats_periods, K0Q.X, K1Q.X, matrix(0, N, N), rho0d*dt, rho1d*dt, dt) # N*J
#print(K1Q.X)

BX <- loads.X.prelim$B
WBXp <- W %*% t(BX)  # N*N
if (rcond(WBXp)<.Machine$double.eps) {
  cat("SINGULARITY -- matrix (W*B_X') is computationally singular\n")
  cat("rcond(WBXp) = ", rcond(WBXp),"\n")
  cat("lamQ = ", diag(K1Q.X), "\n")
  cat("rinfQ = ", rinfQ, "\n")
  stop("singularity error in jsz.loadings\n")
}

Sigma_X <- solve(WBXp, Sigma.cP) %*% solve(t(WBXp))
### (W*BX')\Sigma.cP/(BX*W');

############################################################
# Now with Sigma_X in hand, compute loadings for AX
loads.X <- gaussian.loadings(mats_periods, K0Q.X, K1Q.X, Sigma_X, rho0d*dt, rho1d*dt, dt)
AX <- loads.X$A  # 1*J
BX <- loads.X$B  # N*J

############################################################

############################################################
# Finally, rotate the model to obtain the AcP, BcP loadings.
# (See above for calculation)
WBXp <- W %*% t(BX)  # N*N
WAXp <- W %*% t(AX)  # N*1
WBXpinv <- solve(WBXp) # N*N
BcP <- t(WBXpinv) %*% BX # N*J
#AcP <- AX - t(WAXp) %*% WBXpinv %*% BX # 1*J    # for some reason these are wrong
AcP <- AX %*% t(diag(J) - t(BX)%*% solve(WBXp,W))  # 1*J
############################################################

jsz.loadings <- list(AX=AX, BX=BX, AcP=AcP, BcP=BcP) 
}

########################################################################
########################################################################

jsz.rotation <- function(W, K1Q.X, rinfQ, dt, BX, AX) {
# Inputs:
#   W          : N*J,      vector of portfolio weights to fit without error.
#   K1Q.X      : N*N
#   rinfQ      : scalar,   the long run mean under Q of the annualized short rate
#   dt         : scalar,   length of period in years
#   BX         : N*J  (BX, AX) are optional (saves time)
#   AX         : 1*J
#
# Returns:  [K0Q.cP, K1Q.cP, rho0.cP, rho1.cP]
#   K0Q.cP : N*1
#   K1Q.cP : N*N
#   rho0.cP : scalar
#   rho1.cP : N*1
#
#
# r(t) = rho0.cP + rho1.cP'*cPt
#      = rinfQ + 1'*Xt
#      = 1 period discount rate (annualized)
#
# Under Q:
#   X(t+1) -X_t   =   K1Q.X*X(t)  + eps_X(t+1),   cov(eps_X(t+1)) = Sigma_X
#   cP(t+1) - cP(t) = K0Q.cP + K1Q.cP*X(t) + eps_cP(t+1),  cov(eps_cP(t+1)) = Sigma.cP
##   NOTE:  I redefine K1Q.X to be the coefficient in X(t+1) = beta*X(t) + eps
# Where Sigma_X is chosen to match Sigma.cP 
#
# cPt = W*yt  (cPt N*1, W is N*J)
#     = W*AX' + W*BX'*Xt
#     = WAXp + WBXp*Xt
#
# Delta(cP) = WBXp*Delta(Xt)
#           = WBXp*(K1Q.X*Xt + sqrt(Sigma_X)*eps(t+1))
#           = WBXp*(K1Q.X)*(WBXp\(cPt - WAXp)) + sqrt(Sigma.cP)*eps(t+1)
#           = WBXp*(K1Q.X)/WBXp*cPt - WBXp*(K1Q.X)/WBXp*WAXp] + sqrt(Sigma.cP)*eps(t+1)
#
# rt = rinfQ + 1'*Xt  [annualized 1-period rate]
#    = rinfQ + 1'*(WBXp\(cPt - WAXp))
#    = [rinfQ - 1'*(WBXp\WAXp)] + ((WBXp)'1)'*cPt

N <- nrow(K1Q.X)
WBXp <- W %*% t(BX)
WAXp <- W %*% t(AX)
WBXpinv <- solve(WBXp)

K1Q.cP <- WBXp %*% K1Q.X %*% WBXpinv
K0Q.cP <- (diag(N) - K1Q.cP) %*% WAXp

rho1.cP = t(WBXpinv) %*% rep(1,N)
rho0.cP = rinfQ - t(WAXp) %*% rho1.cP

jsz.rotation <- list(K0Q.cP=K0Q.cP, K1Q.cP=K1Q.cP, rho0.cP=rho0.cP, rho1.cP=rho1.cP)
} # end of jsz.rotation

########################################################################
########################################################################

gaussian.loadings <- function(maturities, K0d, K1d, H0d, rho0d, rho1d, timestep=1) {
# maturities: M*1
# K0d      : N*1
# K1d      : N*1
# H0d      : N*N
# rho0d    : scalar  
# rho1d    : N*1
# timestep : optional argument.
#
# By : N*M
# Ay : 1*M  (faster to not compute with only one output argument)
#
# r(t)   = rho0d + rho1d'Xt
#        = 1 period discount rate
# P(t)   =  price of  t-period zero coupon bond
#        = EQ0[exp(-r0 - r1 - ... - r(t-1)]
#        = exp(A+B'X0)
# yields = Ay + By'*X0
#   yield is express on a per period basis unless timestep is provided.
#   --For example, if the price of a two-year zero is exp(-2*.06)=exp(-24*.005),
#   --and we have a monthly model, the function will return Ay+By*X0=.005
#   --unless timestep=1/12 is provided in which case it returns Ay+By*X0=.06
#
# Where under Q:
#   X(t+1) = K0d + K1d*X(t) + eps(t+1),  cov(eps(t+1)) = H0d
#
# A1 = -rho0d
# B1 = -rho1d
# At = A(t-1) + K0d'*B(t-1) .5*B(t-1)'*H0d*B(t-1) - rho0d
# Bt = 1d'*B(t-1) - rho1d
#
# maturities: 1*M # of periods

M = length(maturities)
N = length(K0d)
Atemp = 0
Btemp = matrix(0,N,1)
Ay = matrix(NA,1,M)
By = matrix(NA,N,M)

curr_mat = 1
K0dp <- t(K0d)
K1dp <- t(K1d)
for (i in 1:maturities[M]) {
    Atemp <- Atemp + K0dp%*%Btemp +.5%*%t(Btemp)%*%H0d%*%Btemp - rho0d
    Btemp <- K1dp%*%Btemp - rho1d
    
    if (i==maturities[curr_mat]) {
        Ay[1,curr_mat] <- -Atemp/maturities[curr_mat]
        By[,curr_mat] <- -Btemp/maturities[curr_mat]
        curr_mat <- curr_mat + 1
    }
}
    
gaussian.loadings <- list(A = Ay/timestep, B = By/timestep)
 # end of gaussian.loadings

} # end of jsz.llk

## older functions - used in dtsm_analysis.r

get.forward.rate <- function(lambda.0, lambda.1, delta.0, delta.1, mu, Phi, Sigma, from.mat, to.mat) {
  loadings <- get.loadings(lambda.0, lambda.1, delta.0, delta.1, mu, Phi, Sigma, to.mat)
  A <- loadings$A
  B <- loadings$B
  Af <- (-A[from.mat]*from.mat + A[to.mat]*to.mat)/(to.mat - from.mat)
  Bf <- (-B[,from.mat]*from.mat + B[,to.mat]*to.mat)/(to.mat - from.mat)
  f.hat <- Af + X%*%Bf
  ts(f.hat, frequency=4, start=c(1990,1))
}

get.loadings <- function(lambda.0, lambda.1, delta.0, delta.1, mu, Phi, Sigma, N) {
  Omega <- Sigma %*% t(Sigma)
  A <- matrix(NA, 1, N)
  B <- matrix(NA, k, N)
  A.tmp <- 0
  B.tmp <- matrix(0,k,1)
  for (n in 1:N) {
    A.tmp <- A.tmp + t(B.tmp) %*% (mu - Sigma %*% lambda.0) + .5*t(B.tmp)%*%Omega%*%B.tmp - delta.0
    B.tmp <- t(Phi - Sigma%*%lambda.1) %*% B.tmp - delta.1
    A[n] <- -A.tmp/n
    B[,n] <- -B.tmp/n
  }
  get.loadings <- list(A=A, B=B)
}

get.yields <- function(lambda.0, lambda.1, delta.0, delta.1, mu, Phi, Sigma, maturities) {
  N <- max(maturities)
  loadings <- get.loadings(lambda.0, lambda.1, delta.0, delta.1, mu, Phi, Sigma, N)
  A <- loadings$A
  B <- loadings$B

  ## create fitted yields
  Y.hat <- t(matrix(rep(A,T), N, T)) + X %*% B
  Y.hat <- Y.hat[, maturities]   # only selected
}

EX.var <- function(Phi, mu, X, H, p) {
  ## returns: EX.var  T by H by k
T <- nrow(X)
k <- ncol(X)
## expected factor paths
EX <- array(NA, c(T,H+1,k))  # expected factor paths
##    E_t X_{t+h},   t=1...T, h=0...H-1
mean.X <- colMeans(X)
if (p==1) {   # VAR(1)
  for (t in 1:T) {
    ## h=0: current values
    EX[t,1,] <- X[t,]
    ## h>0
    #Phi.h <- diag(k)
    for (h in 1:H) {
      #Phi.h <- Phi.h %*% Phi
      #EX[t,1+h,] <- (diag(k) - Phi.h)%*%mean.X + Phi.h %*%X[t,]
      #EX[t,1+h,] <- mean.X + Phi %*% (EX[t,h,]-mean.X)
      EX[t,1+h,] <- (diag(k) - Phi)%*%mean.X + Phi %*% EX[t,h,]
      EX[t,1+h,] <- mu + Phi %*% EX[t,h,]
    }
      
  } # for t
} else {
  ## VAR(2)
  ## note: no forecast at t=1
  for (t in 2:T) {
    ## h=0: current values
    EX[t,1,] <- X[t,]
    ## h=1
    EX[t,2,] <- mean.X + Phi[1:k,1:k] %*% (EX[t,1,]-mean.X) + Phi[1:k,(k+1):(2*k)] %*% (X[t-1,]-mean.X)
    ## h>1
    for (h in 2:H)
      EX[t,h+1,] <- mean.X + Phi[1:k,1:k] %*% (EX[t,h,]-mean.X) + Phi[1:k,(k+1):(2*k)] %*% (EX[t,h-1,]-mean.X)
  } # for t
} # if
EX.var <- EX[,2:(H+1),]
} # function

frn.var <- function(Phi, mu, X, delta.0, delta.1, h.from, h.to, p=1) {
  ## calculate risk-neutral long-rate based on VAR forecast
  ## (ignores Jensen inequality effect; just average of short rate forecasts)
  T <- nrow(X)
  frn.var <- numeric(T)
  frn.var[1] <- NA
  
  EX <- EX.var(Phi, mu, X, H=h.to, p)
  if (h.to>h.from+1) {
    for (t in 1:T)
      frn.var[t] <- 400*(delta.0 + t(delta.1) %*% colMeans(EX[t,h.from:(h.to-1),]))
  } else {
    for (t in 1:T)
      frn.var[t] <- 400*(delta.0 + t(delta.1) %*% EX[t,h.from,])
  }
  frn.var <- ts(frn.var, frequency=freq, start=start.date)
}

jsz.llkQ <- function (yields.o, W, lamQ.X, rinfQ, Sigma.cP, mats) {
# Compute the Q-likelihood for a Gaussian term structure.
# Source "A New Perspective on Gaussian Dynamic Term Structure Models" by Joslin, Singleton and Zhu
#
# INPUTS:
# yields.o   : T*J,  matrix of observed yields (first row are t=0 observations, which likelihood conditions on)
# mats       : 1*J,      maturities in periods
# W          : N*J,      vector of portfolio weights to fit without error.
# lamQ.X     : N*N,      normalized latent-model matrix (does not have to be diagonal, see form below)
# rinfQ      : scalar,   long-run mean under Q, of the annualized short rate
# Sigma.cP   : N*N,      positive definite matrix that is the covariance of innovations to cP
#
# output
# 
# llk        : T*1       time series of -log likelihoods (includes 2-pi constants)
# AcP        : 1*J       yt = AcP' + BcP'*Xt  (yt is J*1 vector)
# BcP        : N*J       AcP, BcP satisfy internal consistency condition that AcP*W' = 0, BcP*W' = I_N
# AX         : 1*J       yt = AX' + BX'*Xt  
# BX         : N*J       Xt is the 'jordan-normalized' latent state
    
T <- nrow(yields.o)
J <- ncol(yields.o)
N <- nrow(W)
cP <- yields.o %*% t(W) # T*N, cP stands for math caligraphic P.

                                        
# First find the loadings for the model:
# yt = AcP' + BcP'*cPt, AcP is 1*J, BcP is N*J
loads <- jsz.loadings(W, diag(lamQ.X), rinfQ, Sigma.cP, mats, dt=1)
##[BcP, AcP, AX, BX]
parP <- jsz.rotation(W, diag(lamQ.X), rinfQ, dt=1, loads$BX, loads$AX)
##[K0Q.cP, K1Q.cP, rho0.cP, rho1.cP]

yields.m =  rep(1,T)%*%loads$AcP + cP %*% loads$BcP # T*J, model-implied yields
yield.errors = yields.o[1:T,] - yields.m[1:T,]; # T*J
square_orthogonal_yield.errors = yield.errors^2; # T*J, but N-dimensional projection onto W is always 0, so effectively (J-N) dimensional

# Compute optimal sigma.e if it is not supplied
sigma.e = sqrt( sum(square_orthogonal_yield.errors)/(T*(J-N)) )
llkQ = .5*rowSums(square_orthogonal_yield.errors)/sigma.e^2 + (J-N)*.5*log(2*pi) + .5*(J-N)*log(sigma.e^2) # 1*T

jsz.llkQ <- list(llkQ=llkQ, AcP=loads$AcP, BcP=loads$BcP, AX=loads$AX, BX=loads$BX, sigma.e=sigma.e, K0Q.cP=parP$K0Q.cP, K1Q.cP = parP$K1Q.cP, rho0.cP=parP$rho0.cP, rho1.cP=parP$rho1.cP, cP=cP)
                 
}

########################################################################
########################################################################

fut.loadings <- function(W, K1Q.X, rinfQ) {
## compute JSZ-loadings for money market futures
# Inputs:
#   W          : N*J,      vector of portfolio weights to fit without error.
#   K1Q.X      : N*N
#   rinfQ      : scalar,   the long run mean under Q of the annualized short rate
#
# Returns:
#   AcP : 1*J
#   BcP : N*J
#   AX  : 1*J
#   BX  : N*J

N <- nrow(K1Q.X)
K0Q.X <- matrix(0,N,1)
rho0d <- rinfQ
rho1d <- rep(1,N)
M = (max(ed.contracts)+1)*NQ

############################################################
# compute loadings for forward rates with horizons 1:M 
loads.X <- gaussian.loadings(1:M, K0Q.X, K1Q.X, matrix(0,N,N), rho0d, rho1d)
## go from yield loadings to forward rate loadings
AfX <- cbind(loads$A[1], (2:M)*loads$A[2:M]-(1:(M-1))*loads$A[1:(M-1)])  # 1*M   
BfX <- cbind(loads$B[,1], rep(1,N)%*%t(2:M) * loads$B[,2:M] - rep(1,N)%*%t(1:(M-1)) * loads$B[,1:(M-1)])  # N*M

########################################################
## from M loadings for daily forward rates, compute J loadings
## of futures contracts BY SIMPLE AVERAGING

## loadings for futures contracts on X
AX <- matrix(NA, 1, J)
BX <- matrix(NA, N, J)

## FF futures
d <- floor(NM/2) ## pretend it's the middle of the month
n2contract <- c(numeric(NM-d), kronecker(seq(1,max(ff.contracts)),rep(1,NM)), numeric(d))
for (j in 1:J1) {
  AX[j] <- mean(AfX[which(n2contract==ff.contracts[j])])
  BX[,j] <- rowMeans(BfX[,which(n2contract==ff.contracts[j])])
}

## ED futures
d <- floor(NQ/2)  ## pretend it's the middle of the quarter
n2contract <- c(numeric(NQ-d), kronecker(seq(1,max(ed.contracts)),rep(1,NQ)), numeric(d))
for (j in 1:J2) {
  AX[J1+j] <- mean(AfX[which(n2contract==ed.contracts[j])])
  BX[,J1+j] <- rowMeans(BfX[,which(n2contract==ed.contracts[j])])
}

## slightly more efficient
#(ind.contract <- matrix(1,J,1)%*%( (n.per-d)+matrix(1:n.per,1,n.per)) + matrix(seq(0,by=n.per,le=J),J,1)%*%matrix(1,1,n.per))

#AX <- matrix(rowMeans(matrix(AfX[ind.contract], J, n.per)), 1, J)
#for (i in 1:N) 
#  BX[i,] <- matrix(rowMeans(matrix(BfX[i,ind.contract], J, n.per)), 1, J)

############################################################
# Finally, rotate the model to obtain the AcP, BcP loadings.
WBXp <- W %*% t(BX)  # N*N
WAXp <- W %*% t(AX)  # N*1
WBXpinv <- solve(WBXp) # N*N
BcP <- t(WBXpinv) %*% BX # N*J
AcP <- AX %*% t(diag(J) - t(BX)%*% solve(WBXp,W))  # 1*J
############################################################

fut.loadings <- list(AX=AX, BX=BX, AcP=AcP, BcP=BcP) 
}

fut.loadings.exact <- function(W, K1Q.X, rinfQ, contracts, n.per) {
  ## compute JSZ-loadings for money market futures
  ## Inputs:
  ##   contracts  : 1*J,      which contracts
  ##   n.per      : scalar,   number of days in contract period
  ##   W          : N*J,      vector of portfolio weights to fit without error.
  ##   K1Q.X      : N*N
  ##   rinfQ      : scalar,   the long run mean under Q of the annualized short rate
  ##
  ## Returns:
  ##   AcP : 1*J
  ##   BcP : N*J
  ##   AX  : 1*J
  ##   BX  : N*J
  
  J <- length(contracts)
  N <- nrow(K1Q.X)
  K0Q.X <- matrix(0,N,1)
  rho0d <- rinfQ
  rho1d <- rep(1,N)
  M = (max(contracts)+1)*n.per
  
############################################################
                                        # compute loadings for forward rates with horizons 1:M 
  loads.X <- gaussian.loadings(1:M, K0Q.X, K1Q.X, matrix(0,N,N), rho0d, rho1d)
  AfX <- loads.X$A  # 1*M
  BfX <- loads.X$B  # N*M
  
########################################################
  ## from M loadings for daily forward rates, compute J loadings
  ## of futures contracts BY SIMPLE AVERAGING
  
  AcP <- matrix(NA, J, n.per)
  BcP <- array(NA, c(N,J,n.per))
  
  for (d in 1:n.per) {
    
    ## loadings for futures contracts on X
    AX <- matrix(NA, 1, J)
    BX <- matrix(NA, N, J)
    
    ## determine mapping from horizon in days to futures contract
    n2contract <- c(numeric(n.per-d), kronecker(seq(J),rep(1,n.per)), numeric(d))
    
    for (j in 1:J) {
      AX[j] <- mean(AfX[n2contract==j])
      BX[,j] <- rowMeans(BfX[,n2contract==j])
    }
    
    ## Finally, rotate the model to obtain the AcP, BcP loadings.
    WBXp <- W %*% t(BX)  # N*N
    WAXp <- W %*% t(AX)  # N*1
    WBXpinv <- solve(WBXp) # N*N
    BcP[,,d] <- t(WBXpinv) %*% BX # N*J
    AcP[,d] <- AX %*% t(diag(J) - t(BX)%*% solve(WBXp,W))  # 1*J
    
  }
  
  fut.loadings.exact <- list(AcP=AcP, BcP=BcP) 
}

yfut.loadings <- function(W, K1Q.X, rinfQ) {
## compute JSZ-loadings for money market futures and yields
# Inputs:
#   W          : N*J,      vector of portfolio weights to fit without error.
#   K1Q.X      : N*N
#   rinfQ      : scalar,   the long run mean under Q of the annualized short rate
#
# Returns:
#   AcP : 1*J
#   BcP : N*J
#   AX  : 1*J
#   BX  : N*J

  ## global variables used:  ff.contracts, ed.contracts, mats, NM, NQ, J1, J2, J3
N <- nrow(K1Q.X)
K0Q.X <- matrix(0,N,1)
rho0d <- rinfQ
rho1d <- rep(1,N)

############################################################
## yields (intercepts NA because COV matrix and thus convexity unknown)
loadsy.X <- gaussian.loadings(mats, K0Q.X, K1Q.X, matrix(NA,N,N), rho0d, rho1d)
AXy <- loadsy.X$A  # 1*J3
BXy <- loadsy.X$B  # N*J3

############################################################
## futures rates

## max horizon needed
M = (max(ed.contracts)+1)*NQ

## compute FORWARD RATE loadings for forward rates with horizons 1:M 
loads <- gaussian.loadings(1:M, K0Q.X, K1Q.X, matrix(NA,N,N), rho0d, rho1d)
## go from yield loadings to forward rate loadings
AfX <- cbind(loads$A[1], (2:M)*loads$A[2:M]-(1:(M-1))*loads$A[1:(M-1)])  # 1*M   
BfX <- cbind(loads$B[,1], rep(1,N)%*%t(2:M) * loads$B[,2:M] - rep(1,N)%*%t(1:(M-1)) * loads$B[,1:(M-1)])  # N*M

## from M loadings for daily forward rates, compute J loadings
## of futures contracts BY SIMPLE AVERAGING

## loadings for futures contracts on X
AXfut <- matrix(NA, 1, J1+J2)
BXfut <- matrix(NA, N, J1+J2)

## FF futures
d <- floor(NM/2) ## pretend it's the middle of the month
n2contract <- c(numeric(NM-d), kronecker(seq(1,max(ff.contracts)),rep(1,NM)), numeric(d))
for (j in 1:J1) {
  AXfut[j] <- mean(AfX[which(n2contract==ff.contracts[j])])
  BXfut[,j] <- rowMeans(BfX[,which(n2contract==ff.contracts[j])])
}

## ED futures
d <- floor(NQ/2)  ## pretend it's the middle of the quarter
n2contract <- c(numeric(NQ-d), kronecker(seq(1,max(ed.contracts)),rep(1,NQ)), numeric(d))
for (j in 1:J2) {
  AXfut[J1+j] <- mean(AfX[which(n2contract==ed.contracts[j])])
  BXfut[,J1+j] <- rowMeans(BfX[,which(n2contract==ed.contracts[j])])
}

############################################################
## combine futures and yields loadings
AX <- cbind(AXfut, AXy)
BX <- cbind(BXfut, BXy)

############################################################
# Finally, rotate the model to obtain the AcP, BcP loadings.
WBXp <- W %*% t(BX)  # N*N
WAXp <- W %*% t(AX)  # N*1
if (rcond(WBXp)<.Machine$double.eps) {
  cat("SINGULARITY -- matrix (W*B_X') is computationally singular\n")
  cat("rcond(WBXp) = ", rcond(WBXp),"\n")
  cat("lamQ = ", diag(K1Q.X), "\n")
  cat("rinfQ = ", rinfQ, "\n")
  stop("singularity error in yfut.loadings\n")
}
WBXpinv <- solve(WBXp) # N*N
BcP <- t(WBXpinv) %*% BX # N*J
AcP <- AX %*% t(diag(J) - t(BX)%*% solve(WBXp,W))  # 1*J
############################################################

yfut.loadings <- list(AX=AX, BX=BX, AcP=AcP, BcP=BcP) 
}

kalman.filter <- function(pars, Y, smoothing=FALSE) {
  ## likelihood evaluation using Kalman filter
  ## for affine Gaussian DTSM -- JSZ canonical form
  ## used in zlb/threefactor/est_jsz_kalman.r
  ##     and zlb/threefactor/zlb.r
  ## list pars needs following variables
  ##  rinfQ
  ##  lamQ 
  ##  Omega
  ##  Phi, mu -- P-dynamics
  ##  sigma.e -- measurement error variance
  
  with(as.list(c(pars,Y=Y,smoothing=smoothing)), {
    ## parameters state equation
    Q <- Omega

    ## parameters measurement equation
    jsz.loads <- jsz.loadings(W, diag(lamQ), rinfQ, Omega, mats)
    G <- as.numeric(jsz.loads$AcP)
    H <- t(jsz.loads$BcP)   # J x N
    R <- diag(J)*sigma.e^2   # J x J

    T <- dim(Y)[1]
    x.tt <- matrix(NA, T, N)
    x.ttm1 <- matrix(NA, T, N)
    P.tt <- array(NA, c(T, N, N))
    P.ttm1 <- array(NA, c(T, N, N))

    ## starting values for x and P
    x.ttm1[1,] <- solve(diag(N)-Phi) %*% mu
    P.ttm1[1,,] <- matrix( solve(diag(N^2) - kronecker(Phi, Phi))%*%as.numeric(Q), N, N);
    
    llk.sum <- 0
    for (t in 1:T) {
      V <- H %*% P.ttm1[t,,] %*% t(H) + R  # MSFE, N x N
      V.inv <- solve(V)
      K <- Phi %*% P.ttm1[t,,] %*% t(H) %*% V.inv  # Kalman gain
      eta <- Y[t,] - G - H %*% x.ttm1[t,]
      llk.sum <- llk.sum -.5*log(det(V)) -.5 * t(eta) %*% V.inv %*% eta

      x.tt[t,] <- x.ttm1[t,] + P.ttm1[t,,] %*% t(H) %*% V.inv %*% eta
      P.tt[t,,] <-  P.ttm1[t,,] - P.ttm1[t,,] %*% t(H) %*% V.inv %*% H %*% P.ttm1[t,,] #P_t|t

      if (t<T) {
        ## update next periods values
        x.ttm1[t+1,] <- mu + Phi %*% x.ttm1[t,] + K %*% eta  # x_t+1|t
        ## should be equal to mu + Phi %*% x.tt[t]
        P.ttm1[t+1,,] <- Phi %*% P.tt[t,,] %*% t(Phi) + Q  # P_t+1|t
      }
    }

    if (smoothing) {
      ## smoothing
      x.tT <- matrix(NA, T, N)
      x.tT[T,] <- x.tt[T,]
      for (t in (T-1):1) {
        J.t <- P.tt[t,,] %*% t(Phi) %*% solve(P.ttm1[t+1,,])
        x.tT[t,] <- x.tt[t,] + J.t %*% (x.tT[t+1,] - x.ttm1[t+1,])
      }
      return(list(llk.sum=llk.sum, x.tT=x.tT, x.tt=x.tt))
    } else {
      return(list(llk.sum=llk.sum, x.tt=x.tt))
    }
  })    
}
