starting.lamQ <- function(N) {
  ## provide N eigenvalues less than one, descending order
  return( sort(runif(N, min=.9, max=1),decreasing=TRUE))
}

get.PhiQ.X <- function(lamQ.X) {
  lamQ.X <- sort(lamQ.X, decreasing=TRUE)
  PhiQ.X <- diag(lamQ.X)
  if (lamQ.X[1]==lamQ.X[2])
    PhiQ.X[1,2] = 1
  if (lamQ.X[2]==lamQ.X[3])
    PhiQ.X[2,3] = 1
#  if (N>3)
#    stop("not implemented for repeated eigenvalues and N>3")
  return(PhiQ.X)
}

Y.fitted <- function(P, W, PhiQ.X, rinfQ) {
  ## obtain fitted values for futures
  loads <- yfut.loadings(W, PhiQ.X, rinfQ)
  Y.hat <- rep(1,nrow(P))%*%loads$AcP + P %*% loads$BcP
}

dY.fitted <- function(dP, P, W, PhiQ.X) {
  ## obtain fitted values for changes in futures rates
  loads <- yfut.loadings(W, PhiQ.X, rinfQ=NA)
  dY.hat <- dP%*%loads$BcP

  return(dY.hat)
}

plotUncondVolCurve <- function(dY, dY.hat, dX, BfX) {
    ## unconditional vol curve
    ## empirical vol curve w/ CI
    vols.dY <- apply(dY, 2, sd)
    alpha <- .05
    CIfact <- (T-2)/qchisq(p=c(1-alpha/2, alpha/2),df=T-2)
    CIfact = CIfact^(.5)
    Lbound = CIfact[1]*vols.dY
    Ubound = CIfact[2]*vols.dY
    ## model-implied volatilities of instruments
    vols.model <- sqrt(diag( var(dY.hat) + diag(J)*sigma.e^2))
    ## model-implied volatilities of forward rate path
    vols.rev <- sqrt(diag(t(BfX) %*% var(dX) %*% BfX))
    ## plot vol curve
    yrange <- range(vols.model,Lbound,Ubound)
    colors <- c("black", "blue", "red", "black") #"#888888","red", "green")
    lwds <- c(2,2,2,1)
    ltys <- c(1,3,1,3)
    dev.new()
    yrange=range(4,9)
    par(mfrow=c(1,2))
    par(mar=c(4,4,2,1))
    plotCI(ed.contracts, vols.dY[Jind.ed], ui=Ubound[Jind.ed], li=Lbound[Jind.ed], ylim=yrange,gap=0,xlab="Quarters",ylab="Basis points")
    lines(ed.contracts, vols.model[Jind.ed], col=colors[2], type="l", lwd=lwds[2], lty=ltys[1])
    title("Eurodollar futures")
    plotCI(mats/n.days, vols.dY[Jind.y], ui=Ubound[Jind.y], li=Lbound[Jind.y], ylim=yrange,gap=0,xlab="Maturity (in years)",ylab="")
    lines(mats/n.days, vols.model[Jind.y], col=colors[2], type="l", lwd=lwds[2], lty=ltys[1])
    title("Yields")
    ##dev.off()
    ## separate graph -- vols of forward rate path
    dev.new()
    plot(mats.rev/n.days, vols.rev, ylim=yrange, type="l", col="blue", lwd=2, lty=1, ylab="", xlab="Horizon (in years)")
    title("volatility of forward path")
}

show.day <- function(t) {
    ## look at revision on specific day
    print("*** analyze specific day ***")
    cat("date = ", dates[t], "\n")
    cat("dP = ", dP[t,], "\n")
    cat("dX = ", dX[t,], "\n")
    print("actual vs. fitted Eurodollar rate changes:")
    print(round( rbind(dY[t, Jind.ed],
                       dX[t,]%*%loads$BX[,Jind.ed]), digi=0))
    print("actual vs. fitted yield changes:")
    print(round( rbind(dY[t, Jind.y],
                       dX[t,]%*%loads$BX[,Jind.y]), digi=0))
    plot(mats.rev/252, dX[t,]%*%BfX)
}

plotPolicySurprises <- function(dY, dY.hat, dX, BfX) {
    ##t = which(dates==20081125)   # initial LSAP announcement
    ## t = which(dates==20080915)   # Lehman
    ## t = which(dates==20041214)   # FOMC, not much news
    dev.new()
    ##png("policy_surprises_pres.png", width=pres.width, height=pres.height, pointsize=24)
    ##postscript("figures/policy_surprises.eps", width=9, height=6, horizontal=FALSE, pointsize=12)
    colors <- c("black", "blue", "red", "black") #"#888888","red", "green")
    lwds <- c(2,2,2,1)
    ltys <- c(1,3,1,3)
    par(mfrow=c(2,3))
    par(mar=c(4,4,2,1))
    ## top panel
    t = which(dates==20050322) ## tightening surprise
    yrange <- range(dY[t,Jind.ed],dY[t,Jind.y],dY.hat[t,Jind.ed])
    plot(ed.contracts,dY[t,Jind.ed], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="Basis points", xlab="", pch=4)
    text(6,10,labels="tightening, 03/22/2005", adj=c(0,0))
    lines(ed.contracts, dY.hat[t,Jind.ed], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    legend("bottomright", c("actual changes","fitted"), col=colors, lwd=lwds,lty=c(NA,ltys[2]), pch=c(4,NA))
    title("Changes in ED futures")
    plot(mats.years,dY[t,Jind.y], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="", xlab="", pch=4)
lines(mats.years, dY.hat[t,Jind.y], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    title("Changes in yields")
    plot(mats.rev/n.days, t(BfX)%*%dX[t,], col="blue", type="l", lwd=2, lty=1, xlab="", ylab="")
    title("Revision")
    ## bottom panel
    t = which(dates==20081201) ## Chairman LSAP speech
    par(mar=c(5,4,1,1))
    yrange <- range(dY[t,Jind.ed],dY[t,Jind.y],dY.hat[t,Jind.ed])
    plot(ed.contracts,dY[t,Jind.ed], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="Basis points", xlab="Quarters", pch=4)
    text(4,-9,labels="Bernanke LSAP, 12/01/2008", adj=c(0,0))
    lines(ed.contracts, dY.hat[t,Jind.ed], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    plot(mats.years,dY[t,Jind.y], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="", xlab="Maturity (in years)", pch=4)
lines(mats.years, dY.hat[t,Jind.y], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    plot(mats.rev/n.days, t(BfX)%*%dX[t,], col="blue", type="l", lwd=2, lty=1, xlab="Horizon (in years)", ylab="")
    ##dev.off()

    ## ## for economic letter
    ## ## export (and plot) fitted surprises on three different days (in one graph)
    ## ## Globals: dates, mats.rev, n.days

    ## data.out <- matrix(NA, length(mats.rev), 4)
    ## data.out[,1] <- mats.rev/n.days
    ## dev.new()
    ## #mp.dates <- c(20081201, 20080122,20070918)
    ## mp.dates <- c(20070918, 20071031, 20071211)
    ## colors <- c("black", "blue", "red", "black") #"#888888","red", "green")
    ## lwds <- c(2,2,2,1)
    ## ltys <- c(1,3,1,3)
    ## t = which(dates==mp.dates[1])   ## Chairman LSAP speech
    ## yrange <- range(t(BfX)%*%dX[t,], 20, -60)
    ## plot(mats.rev/n.days, t(BfX)%*%dX[t,],ylim=yrange, col=colors[1], type="l", lwd=lwds[1], lty=ltys[1], xlab="Horizon (in years)", ylab="Basis points")
    ## data.out[,2] <- t(BfX)%*%dX[t,]
    ## t = which(dates==mp.dates[2])
    ## lines(mats.rev/n.days, t(BfX)%*%dX[t,], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    ## data.out[,3] <- t(BfX)%*%dX[t,]
    ## t = which(dates==mp.dates[3])
    ## lines(mats.rev/n.days, t(BfX)%*%dX[t,], col=colors[3], type="l", lwd=lwds[3], lty=ltys[3])
    ## data.out[,4] <- t(BfX)%*%dX[t,]
    ## write.csv(data.out, file="policy_surprises.csv")

}

plotMacroSurprises <- function(dY, dY.hat, dX, BfX) {
    ## Globals: Jinf.y, Jinf.ed, ed.contracts, mats.years, mats.rev, n.days
    dev.new()
    ##png("figures/macro_surprises_pres.png", width=pres.width, height=pres.height, pointsize=24)
    ##postscript("figures/macro_surprises.eps", width=9, height=6, horizontal=FALSE, pointsize=12)
    colors <- c("black", "blue", "red", "black") #"#888888","red", "green")
    lwds <- c(2,2,2,1)
    ltys <- c(1,3,1,3)
    par(mfrow=c(2,3))
    par(mar=c(4,4,2,1))
    ## good news
    t = which(dates==20090605) ## expected -500 released -345
    ##show.day(t)
    yrange <- range(0,dY[t,Jind.ed],dY[t,Jind.y],dY.hat[t,Jind.ed])
    plot(ed.contracts,dY[t,Jind.ed], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="Basis points", xlab="", pch=4)
    text(3,10,labels="good news, 6/5/2009", adj=c(0,0))
    lines(ed.contracts, dY.hat[t,Jind.ed], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    title("Changes in ED futures")
    plot(mats.years,dY[t,Jind.y], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="", xlab="", pch=4)
    lines(mats.years, dY.hat[t,Jind.y], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    legend("topright", c("actual changes","fitted"), col=colors, lwd=lwds,lty=c(NA,ltys[2]), pch=c(4,NA))
    title("Changes in yields")
    plot(mats.rev/n.days, t(BfX)%*%dX[t,], col="blue", type="l", lwd=2, lty=1, ylab="", xlab="")
    title("Revision")
    ## bad news
    ##t = which(dates==20040806)  ## expected 225 released 32
    ##t = which(dates==20000602)  ## expected 360 released 232
    t = which(dates==20040702)  ## expected 230 release 112
    ##show.day(t)
    par(mar=c(5,4,1,1))
    yrange <- range(0,dY[t,Jind.ed],dY[t,Jind.y],dY.hat[t,Jind.ed])
    plot(ed.contracts,dY[t,Jind.ed], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="Basis points", xlab="Quarters", pch=4)
    text(3,-4,labels="bad news, 7/2/2004", adj=c(0,0))
    lines(ed.contracts, dY.hat[t,Jind.ed], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    plot(mats.years,dY[t,Jind.y], ylim=yrange, col=colors[1], type="p", lwd=lwds[1], ylab="", xlab="Maturity (in years)", pch=4)
    lines(mats.years, dY.hat[t,Jind.y], col=colors[2], type="l", lwd=lwds[2], lty=ltys[2])
    plot(mats.rev/n.days, t(BfX)%*%dX[t,], col="blue", type="l", lwd=2, lty=1, xlab="Horizon (in years)",ylab="")
    ##dev.off()
}

createRegimes <- function() {
    ## news regimes
    ## Globals:  T, dates

    ## load matrix with indicators for announcements
    announce <- as.matrix(read.delim("data/dates_announcements.txt", header=FALSE))
    ## group: FOMC, EMPL, CPI/PPI, RET, OTHERS
    announce.group <- matrix(NA, dim(announce)[1],5)
    ## date and FOMC
    announce.group[,1:2] <- announce[,1:2]
    ## EMPL
    announce.group[,3] <- rowSums(announce[,c(8,12,15)])>0
    ## CPI/PPI
    announce.group[,4] <- rowSums(announce[,c(5,9,16,17)])>0
    ## RET
    announce.group[,5] <- announce[,10]

    regimes <- numeric(T)  ## categorical: exclusive regimes
    for (t in 1:T) {
        ind <- announce.group[,1]==dates[t]
        if (sum(announce.group[ind,2:5])==1) {
            ## if only one group of news
            regimes[t] <- which(announce.group[ind,2:5]==1)
        } else if (sum(announce.group[ind,2:5])==0) {
            ## other days -- no policy/employment/CPI/retail news
            regimes[t] <- 5
        }
        ## days with more than one piece of news have zero
    }

##    cat("How many days have more than one type of news: ",
##        sum(rowSums(announce.group[,2:5])>1), "\n")

    ## export data for Onatski test
    write.csv(cbind(regimes, dX), file="factors_dX.csv")
    write.csv(cbind(regimes, dP), file="factors_dP.csv")
    write.csv(cbind(regimes, dY), file="data_dY.csv")

    return(regimes)
}

createRegimesNew <- function() {
    ## news regimes
    ## Globals:  T, dates

    ## load matrix with indicators for announcements
    announce <- read.delim("data/dates_announcements.txt", header=FALSE)
    announce <- subset(announce, V1>=min(dates) & V2<=max(dates))
    announce <- as.matrix(announce)

    ## group: FOMC, EMPL, CPI/PPI, RET, OTHERS
    announce.group <- matrix(NA, dim(announce)[1],5)
    ## date and FOMC
    announce.group[,1:2] <- announce[,1:2]
    ## EMPL
    announce.group[,3] <- rowSums(announce[,c(8,12,15)])>0
    ## CPI/PPI
    announce.group[,4] <- rowSums(announce[,c(5,9,16,17)])>0
    ## RET
    announce.group[,5] <- announce[,10]

    regimes <- numeric(T)  ## categorical: exclusive regimes
    for (t in 1:T) {
        ind <- announce.group[,1]==dates[t]
        if (sum(ind)==0) {
            regimes[t] <- 7 ## no news
        } else {
            if (sum(announce.group[ind,2:5])==1) {
                ## only one piece of major news
                regimes[t] <- which(announce.group[ind,2:5]==1)
            } else if (sum(announce.group[ind,2:5])>1) {
                ## more than one pieces of major news
                regimes[t] <- 5
            } else if (sum(announce.group[ind,2:5])==0) {
                ## no major news
                regimes[t] <- 6
            }
        }
    }

    print(table(regimes))
    ## export data for Onatski test
    write.csv(cbind(regimes, dX), file="factors_dX.csv")
    write.csv(cbind(regimes, dP), file="factors_dP.csv")
    write.csv(cbind(regimes, dY), file="data_dY.csv")

    return(regimes)
}

KuttnerRegressions <- function(dY, dY.hat, dP, regimes) {
    ## Kuttner type regressions
    ## Globals:
    ##   J1, J2, J3
    ##   loads
    b.ed <- numeric(J2);se.ed <- numeric(J2);R2.ed <- numeric(J2)
    b.y <- numeric(J3);se.y <- numeric(J3);R2.y <- numeric(J3)
    ## model-implied responses
    b.edmod <- numeric(J2); R2.edmod <- numeric(J2)
    b.ymod <- numeric(J3); R2.ymod <- numeric(J3)

    ## sample: FOMC days
    mp.days <- regimes==1

    ## regressions
    ## xdat = policy surprise = one-month ahead fed funds futures contract
    xdat <- dY[mp.days,1]
    xfitdat <- dY.hat[mp.days,1]
    V <- var(dP[mp.days,])
    loads.x <- loads$BcP[,1]

    ## regress ED changes for each contract on policy surprise
    for (j in 1:J2) {
        ydat <- dY[mp.days,J1+j]
        ed.lm <- lm(ydat ~ xdat)
        b.ed[j] <- ed.lm$coef[2]
        se.ed[j] <- sqrt(diag(vcov(ed.lm)))[2]
        R2.ed[j] <- summary(ed.lm)$r.squared
        ## model-implied moments
        loads.y <- loads$BcP[,J1+j]
        b.edmod[j] <- t(loads.x)%*%V%*%loads.y / (t(loads.x)%*%V%*%loads.x + sigma.e^2)
        R2.edmod[j] <- ( t(loads.x)%*%V%*%loads.y )^2 / (t(loads.x)%*%V%*%loads.x + sigma.e^2) / (t(loads.y)%*%V%*%loads.y + sigma.e^2)
    }
    blb.ed <- b.ed-1.96*se.ed
    bub.ed <- b.ed+1.96*se.ed

    ## regress yield changes for each maturity on policy surprise
    for (j in 1:J3) {
        ydat <- dY[mp.days,J1+J2+j]
        yield.lm <- lm(ydat ~ xdat)
        b.y[j] <- yield.lm$coef[2]
        se.y[j] <- sqrt(diag(vcov(yield.lm)))[2]
        R2.y[j] <- summary(yield.lm)$r.squared
        ## model-implied moments
        loads.y <- loads$BcP[,J1+J2+j]
        b.ymod[j] <- t(loads.y)%*%V%*%loads.x / (t(loads.x)%*%V%*%loads.x + sigma.e^2)
        R2.ymod[j] <- ( t(loads.x)%*%V%*%loads.y )^2 / (t(loads.x)%*%V%*%loads.x + sigma.e^2) / (t(loads.y)%*%V%*%loads.y + sigma.e^2)
    }
    blb.y <- b.y-1.96*se.y
    bub.y <- b.y+1.96*se.y

    ## plot results
    dev.new()
    ##png("figures/kuttner_pres.png", width=pres.width, height=pres.height, pointsize=24)
    ##postscript("figures/kuttner.eps", width=6.5, height=4, horizontal=FALSE, pointsize=12)
    par(mfrow=c(1,2))
    par(mar=c(4,4,3,1))
    yrange <- range(0,blb.y,bub.y,blb.ed,bub.ed)
    plotCI(ed.contracts, b.ed, li=blb.ed, ui=bub.ed, ylim=yrange, gap=0, xlab="Quarters",ylab="coefficient / R-squared")
    lines(ed.contracts, R2.ed, col="black", type="p", lwd=2, lty=1, pch=1)
    lines(ed.contracts, b.edmod, col="blue", type="l", lwd=2, lty=1)
    lines(ed.contracts, R2.edmod, col="blue", type="p", lwd=2, lty=1, pch=3)
    title("ED futures")
    par(mar=c(4,3,3,2))
    plotCI(mats.years, b.y, li=blb.y, ui=bub.y, ylim=yrange, gap=0, xlab="Maturity (in years)", ylab="")
    lines(mats.years, R2.y, col="black", type="p", lwd=2, lty=1, pch=1)
    lines(mats.years, b.ymod, col="blue", type="l", lwd=2, lty=1)
    lines(mats.years, R2.ymod, col="blue", type="p", lwd=2, lty=1, pch=3)
    title("Yields")
    ##dev.off()
}

plotVolCurves <- function(dY, dY.hat, dX, BfX, dP, regimes) {
    ## heterogeneity -- vol curves
    ## Globals:
    ##  J2, J3, BfX, N
    ##  Jind.ed, Jind.y, sigma.e
    ## Side effects: creates Omega.P and Omega.X
    R <- max(regimes)
    vols.ed <- matrix(NA, R, J2)
    vols.edmod <- matrix(NA, R, J2)
    vols.y <- matrix(NA, R, J3)
    vols.ymod <- matrix(NA, R, J3)
    vols.rev <- matrix(NA, R, dim(BfX)[2])
    Omega.P <<- array(NA, c(N,N,R))
    Omega.X <<- array(NA, c(N,N,R))
    ## confidence intervals for empirical volatilities
    alpha <- .05
    CIfact <- matrix(NA, R,2)
    ## calculate volatilities in each regimes
    for (r in 1:R) {
        ind.sel <- regimes==r
        vols.ed[r,] <- apply(dY[ind.sel,Jind.ed], 2, sd)
        vols.edmod[r,] <- sqrt(diag(var(dY.hat[ind.sel,Jind.ed]) + diag(J2)*sigma.e^2))
        vols.y[r,] <- apply(dY[ind.sel,Jind.y], 2, sd)
        vols.ymod[r,] <- sqrt(diag(var(dY.hat[ind.sel,Jind.y]) + diag(J3)*sigma.e^2))
        CIfact[r,] <- ((sum(ind.sel)-2)/qchisq(p=c(1-alpha/2, alpha/2),df=sum(ind.sel)-2))^.5
        vols.rev[r,] <- sqrt(diag(t(BfX) %*% var(dX[ind.sel, ]) %*% BfX))
        Omega.X[,,r] <<- var(dX[ind.sel, ])
        Omega.P[,,r] <<- var(dP[ind.sel, ])
    }
    Lbound.ed = CIfact[,1]*vols.ed
    Ubound.ed = CIfact[,2]*vols.ed
    Lbound.y = CIfact[,1]*vols.y
    Ubound.y = CIfact[,2]*vols.y
    ## plot vol curves
    yrange <- range(Lbound.y,Ubound.y,Ubound.ed, vols.ymod, vols.edmod)
    colors <- c("black", "blue", "red", "black") #"#888888","red", "green")
    lwds <- c(2,2,2,1)
    ltys <- c(1,3,1,3)
    regime.desc <- c("FOMC", "Employment", "CPI/PPI", "Retail sales", "Other days")
    ##postscript("figures/volcurves_regimes.eps", width=9, height=6, horizontal=FALSE, pointsize=12)
    dev.new()
    par(mfrow=c(3,R))
    par(mar=c(4,4,2,0.2))
    ## Eurodollars
    yrange <- range(Ubound.ed, Lbound.ed)
    for (r in 1:R) {
        if (r %% R == 1) {ylab.str <- "Basis points"} else {ylab.str <- ""}
        plotCI(ed.contracts,vols.ed[r,], ui=Ubound.ed[r,], li=Lbound.ed[r,], ylim=yrange, gap=0, xlab="Quarters",ylab=ylab.str)
        lines(ed.contracts,vols.edmod[r,], col="blue", type="l", lwd=lwds[2], lty=ltys[1])
        title(regime.desc[r])
    }
    ## yields
    yrange <- range(Ubound.y, Lbound.y)
    for (r in 1:R) {
        if (r %% R == 1) {ylab.str <- "Basis points"} else {ylab.str <- ""}
        plotCI(mats.years,vols.y[r,], ui=Ubound.y[r,], li=Lbound.y[r,], ylim=yrange, gap=0, xlab="Maturity (in years)", ylab=ylab.str)
        lines(mats.years,vols.ymod[r,], col="blue", type="l", lwd=lwds[2], lty=ltys[1])
    }
    ## revision
    yrange <- range(0, vols.rev)
    for (r in 1:R) {
        if (r %% R == 1) {ylab.str <- "Basis points"} else {ylab.str <- ""}
        plot(mats.rev/n.days, vols.rev[r,], ylim=yrange, type="l", col="blue", lwd=2, lty=1, ylab=ylab.str, xlab="Horizon (in years)")
    }
    ##dev.off()

    ## for economic letter -- volatilities on policy days and other days
    ## dev.new()
    ## plot(mats.rev/n.days, sqrt(diag(t(BfX) %*% var(dX[regimes==1, ]) %*% BfX)), ylim=range(0,12), type="l", col=colors[1], lwd=lwds[1], lty=ltys[1], xlab="Horizon (in years)", ylab="basis points")
    ## lines(mats.rev/n.days, sqrt(diag(t(BfX) %*% var(dX[regimes>1, ]) %*% BfX)), col=colors[2], lwd=lwds[2], lty=ltys[2])
    ## legend("bottomright", legend=c("policy", "non-policy"))
    ## data.out <- matrix(NA, length(mats.rev), 3)
    ## data.out[,1] <- mats.rev/n.days
    ## data.out[,2] <- sqrt(diag(t(BfX) %*% var(dX[regimes==1, ]) %*% BfX))
    ## data.out[,3] <- sqrt(diag(t(BfX) %*% var(dX[regimes>0, ]) %*% BfX))
    ## write.csv(data.out, file="volcurves.csv")
}

intraVols <- function(regimes) {
    ## intradaily volatilities
    ## load intraday yield changes (on policy days)
    intra.data <- read.csv("data/gss3_data_feb012005.csv", na.strings=".")
    intra.data$date <- intra.data$year*10000+intra.data$month*100+intra.data$day
    intra.data <- intra.data[19:dim(intra.data)[1],]
    dY.tight <- as.matrix(intra.data[,12:16])  ## tight
    dY.wide <- as.matrix(intra.data[,27:31])  ## wide
    dY.daily <- as.matrix(intra.data[,42:46])  ## daily
    ## data set with ALL days
    cm.data <- read.csv("data/kevin_treasury_yields.csv")
    T2 <- dim(cm.data)[1]
    dY.cm <- rbind(0,cm.data[2:T2,2:6]-cm.data[1:(T2-1),2:6])    ## changes in bps

    ## categorize days
    regime.cm <- numeric(T2)
    for (t in 1:T2) {
        if (any(intra.data$date==cm.data[t,1])) {
            ## GSS policy day
            regime.cm[t] = 1
        } else {
            ## entweder employment news day or other day
            ind.Y <- dates==cm.data[t,1]
            if (any(ind.Y) && regimes[ind.Y]==2) {
                regime.cm[t] <- 2
            } else {
                regime.cm[t] <- 3
            }
        }
    }

    vols.tight <- apply(dY.tight, 2, sd)*100
    vols.wide <- apply(dY.wide, 2, sd)*100
    vols.daily <- apply(dY.daily, 2, sd)*100
    vols.cm.all <- apply(dY.cm, 2, sd)*100

    results <- as.matrix(rbind(vols.tight,vols.wide,vols.daily, vols.cm.all))
    colnames(results) <- c("3m","6m","2y","5y","10y")

    for (i in 1:3) {
        vols.regime <- apply(dY.cm[regime.cm==i,], 2, sd)*100
        results <- rbind(results, vols.regime)
        rownames(results)[i+4] <- paste("regime ", i)
    }

    print(round(results, digits=4))

    mats2 <- c(3,6,24,60,120)

    postscript("figures/volcurves_intra.eps", width=6.5, height=4, horizontal=FALSE, pointsize=12)
    ##dev.new()
    yrange=range(0,vols.tight,vols.wide,vols.daily,vols.cm.all)
    par(mfrow=c(1,2))
    par(mar=c(4.2,4,4,2))
    ltys <- c(1,2,3)
    lwds <- c(2, 2, 2)
    colors <- c("black","gray","blue")

    plot(mats2/12, vols.tight, ylim=yrange,type="o",col=colors[1],lty=ltys[1],lwd=lwds[1],xlab="Maturity (in years)",ylab="Basis points",pch=19)
    lines(mats2/12, vols.wide, type="b", lwd=lwds[2], lty=ltys[2],col=colors[2],pch=2)
    lines(mats2/12, vols.daily, type="b", lwd=lwds[3], lty=ltys[3],col=colors[3],pch=3)
    legend("bottomright", c("tight window","wide window","daily window"), col=colors, lwd=lwds,lty=ltys,pch=c(19,2,3))
    title("Policy days")

    plot(mats2/12, vols.cm.all, ylim=yrange,type="o",lty=ltys[1],pch=19,lwd=lwds[1],xlab="Maturity (in years)",ylab="")
    title("All days")
    dev.off()
}

AnnouncementEffects <- function(dY, dY.hat, dX, BfX, dP) {
    ## macro regressions
    ## Globals:   T, dates, J1, J2, J3
    newsDesc = c('Payrolls',
    'Unemployment rate',
    'Hourly earnings',
    'Core CPI',
    'Core PPI',
    'Retail sales')

    raw.news <- read.csv("data/fed_raw_news_csv.csv")

    nnews <- (ncol(raw.news)-1)/2
    cat("Announcement data -- number of releases:", nnews, "\n")
    cols.release <- seq(3,ncol(raw.news),2)
    cols.consensus <- cols.release-1
    data.news <- cbind(raw.news[,1], raw.news[,cols.release]-raw.news[,cols.consensus])
    data.macro <- data.news

    ## create matrix with correct number of obs
    news <- matrix(0, T, 6)
    for (t in 1:T) {
        ind.news <- data.macro[,1]==dates[t]
        if (any(ind.news))
            news[t,] <- as.numeric(data.macro[ind.news, 1+c(6, 10, 13, 3, 7, 8)])
        ## nonfarm payroll, unempl, hourly earnings, core CPI, core PPI, retail sales (see data desc)
    }
    news[is.na(news)] <- 0

    ## standardize news to have unit SD
    for (iNews in 1:6) {
        sd.news <- sd(news[ news[,iNews]!=0 ,iNews])
        cat('standard deviation -- ', newsDesc[iNews], ': ', sd.news, '\n')
        news[,iNews] <- news[,iNews]/sd.news
    }

    ## YIELDS
    b.y <- matrix(NA, 6, J3);se.y <- matrix(NA, 6, J3)
    b.yhat <- matrix(NA, 6, J3);se.yhat <- matrix(NA, 6, J3)
    for (j in 1:J3) {
        ## regress actual rate changes on news
        yield.lm <- lm(dY[,J1+J2+j] ~ news)
        b.y[,j] <- yield.lm$coef[2:7]
        se.y[,j] <- sqrt(diag(vcovHC(yield.lm,type="HC0")))[2:7]
        ## regress fitted rate changes on news
        yhat.lm <- lm(dY.hat[,J1+J2+j] ~ news)
        b.yhat[,j] <- yhat.lm$coef[2:7]
        se.yhat[,j] <- sqrt(diag(vcovHC(yhat.lm,type="HC0")))[2:7]
    }
    blb.y <- b.y-1.96*se.y
    bub.y <- b.y+1.96*se.y
    blb.yhat <- b.yhat-1.96*se.yhat
    bub.yhat <- b.yhat+1.96*se.yhat

    ## EURODOLLAR FUTURES
    b.ed <- matrix(NA, 6, J2)
    se.ed <- matrix(NA, 6, J2)
    b.edhat <- matrix(NA, 6, J2)
    se.edhat <- matrix(NA, 6, J2)
    for (j in 1:J2) {
        ## regress actual rate changes on news
        ed.lm <- lm(dY[,J1+j] ~ news)
        b.ed[,j] <- ed.lm$coef[2:7]
        se.ed[,j] <- sqrt(diag(vcovHC(ed.lm,type="HC0")))[2:7]
        ## regress fitted rate changes on news
        edhat.lm <- lm(dY.hat[,J1+j] ~ news)
        b.edhat[,j] <- edhat.lm$coef[2:7]
        se.edhat[,j] <- sqrt(diag(vcovHC(edhat.lm,type="HC0")))[2:7]
    }
    blb.ed <- b.ed-1.96*se.ed
    bub.ed <- b.ed+1.96*se.ed
    blb.edhat <- b.edhat-1.96*se.edhat
    bub.edhat <- b.edhat+1.96*se.edhat

    ## FORWARD RATES
    b.f <- matrix(NA, 6, 8);se.f <- matrix(NA, 6, 8)
    b.fhat <- matrix(NA, 6, 8);se.fhat <- matrix(NA, 6, 8)
    for (j in 1:8) {
        ## regress actual rate changes on news
        df <- (j+1)*dY[,J-8+j]-j*dY[,J-9+j]
        f.lm <- lm(df ~ news)
        b.f[,j] <- f.lm$coef[2:7]
        se.f[,j] <- sqrt(diag(vcovHC(f.lm,type="HC0")))[2:7]
        ## regress fitted rate changes on news
        df.hat <- (j+1)*dY.hat[,J-8+j]-j*dY.hat[,J-9+j]
        fhat.lm <- lm(df.hat ~ news)
        b.fhat[,j] <- fhat.lm$coef[2:7]
        se.fhat[,j] <- sqrt(diag(vcovHC(fhat.lm,type="HC0")))[2:7]
    }
    blb.f <- b.f-1.96*se.f
    bub.f <- b.f+1.96*se.f
    blb.fhat <- b.fhat-1.96*se.fhat
    bub.fhat <- b.fhat+1.96*se.fhat

    ## compare long-horizon (0-to-10-year) forward rate responses
    out.matrix <- matrix(NA,7,6)
    rownames(out.matrix) <- c("int",newsDesc)
    colnames(out.matrix) <- c("actual", "SE", "SE(White)", "model-fitted", "SE", "SE(White)")
    ## fitted forward rate (for table in paper)
    df.hat <- 10*dY.hat[,J]-9*dY.hat[,J-1]
    f.lm <- lm(df.hat ~ news)
    f.resp <- f.lm$coef[2:7]
    f.se <- sqrt(diag(vcovHC(f.lm,type="HC0")))[2:7]
    print("*** macro regression: nine-to-ten-year forward rate ***")
    print(summary(f.lm))
    out.matrix[,4] <- f.lm$coef
    out.matrix[,5] <- sqrt(diag(vcov(f.lm)))
    out.matrix[,6] <- sqrt(diag(vcovHC(f.lm,type="HC0")))
    ## actual forward rate
    df <- 10*dY[,J]-9*dY[,J-1]
    f.lm <- lm(df ~ news)
    out.matrix[,1] <- f.lm$coef
    out.matrix[,2] <- sqrt(diag(vcov(f.lm)))
    out.matrix[,3] <- sqrt(diag(vcovHC(f.lm,type="HC0")))
    print("nine-to-ten-year forward rate response")
    print(round(out.matrix,digi=2))

    ## responses of principal components
    betaP <- matrix(NA, 3, 6)
    lm.P1 <- lm(dP[,1] ~ news)
    lm.P2 <- lm(dP[,2] ~ news)
    lm.P3 <- lm(dP[,3] ~ news)
    betaP[1,] <- lm.P1$coef[2:7]
    betaP[2,] <- lm.P2$coef[2:7]
    betaP[3,] <- lm.P3$coef[2:7]
    ## responses of Jordan-normalized factors
    betaX <- matrix(NA, 3, 6)
    lm.X1 <- lm(dX[,1] ~ news)
    lm.X2 <- lm(dX[,2] ~ news)
    lm.X3 <- lm(dX[,3] ~ news)
    betaX[1,] <- lm.X1$coef[2:7]
    betaX[2,] <- lm.X2$coef[2:7]
    betaX[3,] <- lm.X3$coef[2:7]
    ## note: betaX = t(t(betaP) %*% t(solve(W %*% t(loads$BX))) )

    ## scaling factor for first PC
    a <- mean(loads$BcP[1,])
    (level.resp <- betaP[1,]*a)
    (level.se <- sqrt(diag(vcovHC(lm.P1,type="HC0")))[2:7]*abs(a))
    (lr.resp <- betaX[1,])
    (lr.se <- sqrt(diag(vcovHC(lm.X1, type="HC0")))[2:7])

    ## table for paper
    out.matrix <- matrix(NA, 12,6)
    ind <- c(1,3,5,7,9,11)
    rownames(out.matrix) <- rep("",12)
    rownames(out.matrix)[ind] <- newsDesc
    colnames(out.matrix) <- c("ED4", "2y yld", "10y yld", "9-to-10 F", "$P^1_t$","$X^1_t$")
    out.matrix[ind,1] <- b.edhat[,3]  ## ED4
    out.matrix[ind+1,1] <- se.edhat[,3]  ## ED4
    out.matrix[ind,2] <- b.yhat[,4]  ## 2y
    out.matrix[ind+1,2] <- se.yhat[,4]  ## 2y
    out.matrix[ind,3] <- b.yhat[,J3]  ## 10y
    out.matrix[ind+1,3] <- se.yhat[,J3]  ## 10y
    out.matrix[ind,4] <- f.resp
    out.matrix[ind+1,4] <- f.se
    out.matrix[ind,5] <- level.resp
    out.matrix[ind+1,5] <- level.se
    out.matrix[ind,6] <- lr.resp
    out.matrix[ind+1,6] <- lr.se
    print(round(out.matrix, digi=2))

    ## confidence bands
    ## calculate variance-covariance matrix for multivariate regression (Hayashi 4.5.23)
    eps.hat <- cbind(lm.X1$residuals, lm.X2$residuals, lm.X3$residuals)
    Sig.hat <- T^-1*t(eps.hat)%*%eps.hat
    X <- cbind(1,news)
    Avar <- kronecker(Sig.hat, solve(t(X)%*%X/T))
    ## now sqrt(Avar[2,2]/T) corresponds to SE on news1 in level.lm

    ## ## plot responses of yields
    ## dev.new()
    ## ##png("macronews_yields_pres.png", width=pres.width, height=pres.height, pointsize=24)
    ## par(mfrow=c(2,3)) ## for screen/presentation
    ## ##postscript("figures/macronews_yields.eps", width=7, height=10, horizontal=FALSE, pointsize=12)
    ## ##par(mfrow=c(3,2)) ## for paper
    ## par(mar=c(4,4,3,1))
    ## for (i in 1:6) {
    ##     yrange <- range(0,blb.y[i,],bub.y[i,], blb.yhat[i,], bub.yhat[i,], blb.ed[i,],bub.ed[i,])
    ##     if (i>3) {xlab.str <- "Maturity (in years)"} else {xlab.str <- ""}
    ##     if (i %% 3 == 1) {ylab.str <- "Basis points"} else {ylab.str <- ""}
    ##     plotCI(mats.years, b.y[i,], li=blb.y[i,], ui=bub.y[i,], ylim=yrange, ylab=ylab.str, xlab=xlab.str,gap=0)
    ##     lines(mats.years, b.yhat[i,],col="blue", type="l", lwd=2, lty=1)
    ##     lines(mats.years, blb.yhat[i,], col="blue", type="l", lwd=2, lty=3)
    ##     lines(mats.years, bub.yhat[i,], col="blue", type="l", lwd=2, lty=3)
    ##     abline(h=0, lty=3)
    ##     title(newsDesc[i])
    ## }
    ## ##dev.off()

    ## plot response of Eurodollar futures
    dev.new()
    ##png("macronews_ED_pres.png", width=pres.width, height=pres.height, pointsize=24)
    par(mfrow=c(2,3))  ## for screen/presentation
    ##postscript("figures/macronews_ED.eps", width=6, height=8, horizontal=FALSE, pointsize=12)
    ##par(mfrow=c(3,2)) ## for paper
    par(mar=c(4,4,3,1))
    for (i in 1:6) {
        yrange <- range(0,blb.y[i,],bub.y[i,], blb.yhat[i,], bub.yhat[i,], blb.ed[i,],bub.ed[i,])
        if (i>4) {xlab.str <- "Quarters"} else {xlab.str <- ""}
        if (i %% 2 == 1) {ylab.str <- "Basis points"} else {ylab.str <- ""}
        plotCI(ed.contracts, b.ed[i,], li=blb.ed[i,], ui=bub.ed[i,], ylim=yrange, ylab=ylab.str, xlab=xlab.str,gap=0)
        lines(ed.contracts, b.edhat[i,], col="blue", type="l", lwd=2, lty=1)
        lines(ed.contracts, blb.edhat[i,], col="blue", type="l", lwd=2, lty=3)
        lines(ed.contracts, bub.edhat[i,], col="blue", type="l", lwd=2, lty=3)
        abline(h=0, lty=3)
        title(newsDesc[i])
    }
    ##dev.off()

    ## plot response of forward rates -- payrolls only
    dev.new()
    ##png("macronews_forwards_payrolls.png", width=pres.width, height=pres.height, pointsize=24)
    ##postscript("figures/macronews_forwards_payrolls.eps", width=7, height=5, horizontal=FALSE, pointsize=12)
    i <- 1
    yrange <- range(0,blb.f[i,],bub.f[i,], blb.yhat[i,], bub.yhat[i,], blb.ed[i,],bub.ed[i,])
    xlab.str <- "Maturity (in years)"
    ylab.str <- "Basis points"
    plotCI(3:10, b.f[i,], li=blb.f[i,], ui=bub.f[i,], ylim=yrange, ylab=ylab.str, xlab=xlab.str,gap=0)
    lines(3:10, b.fhat[i,],col="blue", type="l", lwd=2, lty=1)
    lines(3:10, blb.fhat[i,], col="blue", type="l", lwd=2, lty=3)
    lines(3:10, bub.fhat[i,], col="blue", type="l", lwd=2, lty=3)
    abline(h=0, lty=3)
    ##dev.off()

    ## plot response of Q-expectations
    dev.new()
    ##png("macronews_rev_pres.png", width=pres.width, height=pres.height, pointsize=24)
    ##par(mfrow=c(2,3)) # for screen/presentation
    ##postscript("figures/macronews_rev.eps", width=6, height=8, horizontal=FALSE, pointsize=12)
    par(mfrow=c(3,2)) # for paper
    par(mar=c(4,4,3,1))
    for (i in 1:6) {
        if (i %% 3 == 1) {ylab.str <- "Basis points"} else {ylab.str <- ""}
        if (i>3) {xlab.str <- "Horizon (in years)"} else {xlab.str <- ""}
        rev.resp <- betaX[,i]%*%BfX
        ## confidence bands from multivariate regression
        ind <- 1+i+c(0,7,14)  ## index of the three coefficients in 21x1 stacked vector
        bX.cov <- Avar[ind,ind]/T
        bhat.cov <- t(BfX) %*% bX.cov %*% BfX
        bhat.ub <- rev.resp+1.96*sqrt(diag(bhat.cov))
        bhat.lb <- rev.resp-1.96*sqrt(diag(bhat.cov))
        yrange <- range(0, rev.resp,lr.resp[i]+2*lr.se[i],lr.resp[i]-2*lr.se[i],bhat.ub, bhat.lb)
        plot(mats.rev/n.days, rev.resp, ylim=yrange, ylab=ylab.str,xlab=xlab.str, type="l", lwd=2, col="blue")
        abline(h=0, lty=3)
        lines(mats.rev/n.days, bhat.ub, type="l", col="blue", lwd=1, lty=2)
        lines(mats.rev/n.days, bhat.lb, type="l", col="blue", lwd=1, lty=2)
        title(newsDesc[i])
    }
    ##dev.off()

    ## for Economic Letter -- plot only payrolls and CPI
    ## data.out <- matrix(NA, length(mats.rev), 7)
    ## data.out[,1] <- mats.rev/n.days
    ## icol <- 1
    ## for (i in c(1,4)) {
    ##   rev.resp <- betaX[,i]%*%BfX

    ##   ## confidence bands from multivariate regression
    ##   ind <- 1+i+c(0,7,14)  ## index of the three coefficients in 21x1 stacked vector
    ##   bX.cov <- Avar[ind,ind]/T
    ##   bhat.cov <- t(BfX) %*% bX.cov %*% BfX
    ##   bhat.ub <- rev.resp+1.96*sqrt(diag(bhat.cov))
    ##   bhat.lb <- rev.resp-1.96*sqrt(diag(bhat.cov))

    ##   data.out[,icol+1] <- rev.resp
    ##   data.out[,icol+2] <- bhat.lb
    ##   data.out[,icol+3] <- bhat.ub

    ##   icol <- icol+3
    ## }
    ## write.csv(data.out, file="macro_resp.csv")
}

analyzePolicyInertia <- function(dX, BfX, regimes) {
    ## policy inertia --  relative changes
    ## Globals: n.days, N, PhiQ.X
    summarizeRatio <- function(num, denom) {
        num <- round(num, digi=0); denom <- round(denom, digi=0)
        ratio <- num/denom;
        ratio <- ratio[is.finite(ratio)]
        cat("median = ", round(median(ratio),digi=2), "\n",
            "mean   = ", round(mean(ratio), digi=2), "\n",
            "  SE   =(", round(sd(ratio)/sqrt(length(ratio)), digi=2), ")\n", sep="")
        ## print(t.test(ratio, mu=1,alternative="greater"))
    }
    ind.sel <- regimes>=2 & regimes<=4
    cat("Policy inertia: relative rate changes on", sum(ind.sel), "macro announcement days\n")

    ## short-rate expectations
    cat("Short-rate expectations\n")
    for (h1 in c(.25,.5,.75)*12) {
        h2 <- h1 + .25*12
        cat("horizons: ", c(h1,h2)/12, "\n")
        df2 <- dX%*%BfX[,h2]
        df1 <- dX%*%BfX[,h1]
        num <- df2[ind.sel]
        denom <- df1[ind.sel]
        summarizeRatio(num, denom)
        #print(cbind(num,denom))
    }

    ## yields - forward rate
    ## THIS ASSUMES THAT YIELD MATURITIES ARE even-spaced
    cat("Fitted yields/forward rates:\n")
    for (j in 2:3) {
        cat("Fitted yield ", j, " vs. ", j-1, "\n", sep="")
        num <- j*dY.hat[ind.sel,J1+J2+j]-(j-1)*dY.hat[ind.sel,J1+J2+j-1]
        if (j-1==1) {
            denom <- dY.hat[ind.sel,J1+J2+j-1]
        } else {
            denom <- (j-1)*dY.hat[ind.sel,J1+J2+j-1]-(j-2)*dY.hat[ind.sel,J1+J2+j-2]
        }
        summarizeRatio(num, denom)
        cat("Actual yield ", j, " vs. ", j-1, "\n", sep="")
        num <- j*dY[ind.sel,J1+J2+j]-(j-1)*dY[ind.sel,J1+J2+j-1]
        if (j-1==1) {
            ## denominator is 3-month yield
            denom <- dY[ind.sel,J1+J2+j-1]
        } else {
            denom <- (j-1)*dY[ind.sel,J1+J2+j-1]-(j-2)*dY[ind.sel,J1+J2+j-2]
        }
        summarizeRatio(num, denom)
    }

    ## ED
    cat("Eurodollar futures:\n")
    for (j in 2:3) {
        cat("fitted ED", j+1, " vs. ED", j, "\n", sep="")
        num <- dY.hat[ind.sel,J1+j]
        denom <- dY.hat[ind.sel,J1+j-1]
        summarizeRatio(num, denom)
        cat("actual ED", j+1, " vs. ED", j, "\n", sep="")
        num <- dY[ind.sel,J1+j]
        denom <- dY[ind.sel,J1+j-1]
        summarizeRatio(num, denom)
    }
}

compareRegimes <- function(Omega.P, Omega.X, BfX) {
    ## differences in co-movement
    corrRev <- function(r, i, j) {
        cov.ij <- t(BfX[,i]) %*% Omega.X[,,r] %*% BfX[,j]
        var.i <- t(BfX[,i]) %*% Omega.X[,,r] %*% BfX[,i]
        var.j <- t(BfX[,j]) %*% Omega.X[,,r] %*% BfX[,j]
        return(cov.ij/sqrt(var.i*var.j))
    }
    R <- dim(Omega.P)[3]
    print("correlations and energy contents across regimes")
    tbl <- matrix(NA, R, 7)
    colnames(tbl) <- c("corr(3m,5y)", "corr(l,s)", "corr(l,c)", "corr(s,c)", "energy(PC1)", "energy(PC2)", "energy(PC3)")
    for (r in 1:R) {
        V <- Omega.P[,,r]
        Rmat <- cov2cor(V)
        eigs <- eigen(V)$values
        tbl[r, 1] <- corrRev(r, 3, 60)
        tbl[r, 2:4] <- Rmat[lower.tri(Rmat)]
        tbl[r, 5:7] <- eigs/sum(eigs)*100
    }
    print(round(tbl, digi=2))
}

testRegimes <- function(dP, regimes) {
    ## test for homogeneity of variances across regimes
    ## do not consider days with more than one major news
    ## (regimes == 0)
    dP <- dP[regimes>0,]
    regimes <- regimes[regimes>0]
    ## indicator for monetary policy days
    mp.days <- regimes==1

    p <- numeric(N)
    cat("*** Testing homogeneity of variances\n")

    cat("(i) comparing policy to macro days\n")
    for (i in 1:N)
        p[i] <- bartlett.test(dP[regimes<5,i], mp.days[regimes<5])$p.value
    cat(sprintf("level: %6.4f, slope: %6.4f, curve: %6.4f \n", p[1], p[2], p[3]))

    cat("(ii) comparing policy to all three macro regimes\n")
    for (i in 1:N)
        p[i] <- bartlett.test(dP[regimes<5,i], regimes[regimes<5])$p.value
    cat(sprintf("level: %6.4f, slope: %6.4f, curve: %6.4f \n", p[1], p[2], p[3]))

    cat("(iii) comparing policy to employment report days\n")
    for (i in 1:N)
        p[i] <- bartlett.test(dP[regimes<3,i], regimes[regimes<3])$p.value
    cat(sprintf("level: %6.4f, slope: %6.4f, curve: %6.4f \n", p[1], p[2], p[3]))

    cat("(iv) comparing policy to CPI news days\n")
    sel <- regimes==1 | regimes==3
    for (i in 1:N)
        p[i] <- bartlett.test(dP[sel,i], regimes[sel])$p.value
    cat(sprintf("level: %6.4f, slope: %6.4f, curve: %6.4f \n", p[1], p[2], p[3]))

    cat("(v) comparing policy to retail sales days\n")
    sel <- regimes==1 | regimes==4
    for (i in 1:N)
        p[i] <- bartlett.test(dP[sel,i], regimes[sel])$p.value
    cat(sprintf("level: %6.4f, slope: %6.4f, curve: %6.4f \n", p[1], p[2], p[3]))

    cat("(vi) comparing macro news regimes\n")
    sel <- regimes>1 | regimes<5
    for (i in 1:N)
        p[i] <- bartlett.test(dP[sel,i], regimes[sel])$p.value
    cat(sprintf("level: %6.4f, slope: %6.4f, curve: %6.4f \n", p[1], p[2], p[3]))

    cat("(vii) comparing each regime to 'no/other news'\n")
    for (j in 1:4) {
        sel <- regimes==j | regimes==max(regimes)
        for (i in 1:N)
            p[i] <- bartlett.test(dP[sel,i], regimes[sel])$p.value
        cat("regime", j, "\n")
        cat(sprintf("level: %6.4f, slope: %6.4f, curve: %6.4f \n", p[1], p[2], p[3]))
    }

    ## cat("critical value: ", qchisq(.05, 1, lower.tail=FALSE), "\n")

    cat("*** Testing homogeneity of covariances btw policy and macro\n")
    print(BoxMTest(dP[regimes<5,], as.factor(mp.days[regimes<5])))
}

BoxMTest <- function(X, cl, alpha=0.05) {
    ## source: http://www.public.iastate.edu/~maitra/stat501/Rcode/BoxMTest.R
    ## # Box's M-test for testing homogeneity of covariance matrices
    ## #
    ## # Written by Andy Liaw (2004) converted from Matlab
    ## # Andy's note indicates that he has left the original Matlab comments intact
    ## #
    ## #
    ## # Slight clean-up and fix with corrected documentation provided by Ranjan Maitra (2012)
    ## #
    ## Multivariate Statistical Testing for the Homogeneity of Covariance
    ## Matrices by the Box's M.
    ##
    ## Syntax: function [MBox] = BoxMTest(X,alpha)
    ##
    ## Inputs:
    ## X - data matrix (Size of matrix must be n-by-p;  # RM changed
    ## variables=column 1:p).
    ## alpha - significance level (default = 0.05).
    ## Output:
    ## MBox - the Box's M statistic.
    ## Chi-sqr. or F - the approximation statistic test.
    ## df's - degrees' of freedom of the approximation statistic test.
    ## P - observed significance level.
    ##
    ## If the groups sample-size is at least 20 (sufficiently large),
    ## Box's M test takes a Chi-square approximation; otherwise it takes
    ## an F approximation.
    ##
    ## Example: For a two groups (g = 2) with three independent variables
    ## (p = 3), we are interested in testing the homogeneity of covariances
    ## matrices with a significance level = 0.05. The two groups have the
    ## same sample-size n1 = n2 = 5.
    ## Group
    ## ---------------------------------------
    ## 1 2
    ## ---------------------------------------
    ## x1 x2 x3 x1 x2 x3
    ## ---------------------------------------
    ## 23 45 15 277 230 63
    ## 40 85 18 153 80 29
    ## 215 307 60 306 440 105
    ## 110 110 50 252 350 175
    ## 65 105 24 143 205 42
    ## ---------------------------------------
    ##
    ##
    ## Not true for R
    ##
    ##
    ## Total data matrix must be:
    ## X=[1 23 45 15;1 40 85 18;1 215 307 60;1 110 110 50;1 65 105 24;
    ## 2 277 230 63;2 153 80 29;2 306 440 105;2 252 350 175;2 143 205 42];
    ##
    ##
    ## Calling on Matlab the function:
    ## MBoxtest(X,0.05)
    ##
    ## Answer is:
    ##
    ## ------------------------------------------------------------
    ## MBox F df1 df2 P
    ## ------------------------------------------------------------
    ## 27.1622 2.6293 6 463 0.0162
    ## ------------------------------------------------------------
    ## Covariance matrices are significantly different.
    ##

    ## Created by A. Trujillo-Ortiz and R. Hernandez-Walls
    ## Facultad de Ciencias Marinas
    ## Universidad Autonoma de Baja California
    ## Apdo. Postal 453
    ## Ensenada, Baja California
    ## Mexico.
    ## atrujo_at_uabc.mx
    ## And the special collaboration of the post-graduate students of the 2002:2
    ## Multivariate Statistics Course: Karel Castro-Morales,
    ## Alejandro Espinoza-Tenorio, Andrea Guia-Ramirez, Raquel Muniz-Salazar,
    ## Jose Luis Sanchez-Osorio and Roberto Carmona-Pina.
    ## November 2002.
    ##
    ## To cite this file, this would be an appropriate format:
    ## Trujillo-Ortiz, A., R. Hernandez-Walls, K. Castro-Morales,
    ## A. Espinoza-Tenorio, A. Guia-Ramirez and R. Carmona-Pina. (2002).
    ## MBoxtest: Multivariate Statistical Testing for the Homogeneity of
    ## Covariance Matrices by the Box's M. A MATLAB file. [WWW document].
    ## URL http://www.mathworks.com/matlabcentral/fileexchange/loadFile.do?objectId=2733&objectType=FILE
    ##
    ## References:
    ##
    ## Stevens, J. (1992), Applied Multivariate Statistics for Social Sciences.
    ## 2nd. ed., New-Jersey:Lawrance Erlbaum Associates Publishers. pp. 260-269.

    if (alpha <= 0 || alpha >= 1)
        stop('significance level must be between 0 and 1')
    g = nlevels(cl) ## Number of groups.
    n = table(cl) ## Vector of groups-size.
    N = nrow(X)
    p = ncol(X)
    bandera = 2
    if (any(n >= 20))
        bandera = 1
    ## Partition of the group covariance matrices.

    print(covList <- tapply(as.matrix(X), rep(cl, ncol(X)), function(x, nc) cov(matrix(x, nc = nc)),
                      ncol(X)))
    deno = sum(n) - g
    suma = array(0, dim=dim(covList[[1]]))
    for (k in 1:g)
        suma = suma + (n[k] - 1) * covList[[k]]
    Sp = suma / deno ## Pooled covariance matrix.
    Falta=0
    for (k in 1:g)
        Falta = Falta + ((n[k] - 1) * log(det(covList[[k]])))

    MB = (sum(n) - g) * log(det(Sp)) - Falta ## Box's M statistic.
    suma1 = sum(1 / (n[1:g] - 1))
    suma2 = sum(1 / ((n[1:g] - 1)^2))
    C = (((2 * p^2) + (3 * p) - 1) / (6 * (p + 1) * (g - 1))) *
        (suma1 - (1 / deno)) ## Computing of correction factor.
    if (bandera == 1)
    {
        X2 = MB * (1 - C) ## Chi-square approximation.
        v = as.integer((p * (p + 1) * (g - 1)) / 2) ## Degrees of freedom.
        ## Significance value associated to the observed Chi-square statistic.
        P = pchisq(X2, v, lower=FALSE)  #RM: corrected to be the upper tail
        cat('------------------------------------------------\n');
        cat(' MBox Chi-sqr. df P\n')
        cat('------------------------------------------------\n')
        cat(sprintf("%10.4f%11.4f%12.i%13.4f\n", MB, X2, v, P))
        cat('------------------------------------------------\n')
        if (P >= alpha) {
            cat('Covariance matrices are not significantly different.\n')
        } else {
            cat('Covariance matrices are significantly different.\n')
        }
        return(list(MBox=MB, ChiSq=X2, df=v, pValue=P))
    }
    else
    {
        ## To obtain the F approximation we first define Co, which combined to
        ## the before C value are used to estimate the denominator degrees of
        ## freedom (v2); resulting two possible cases.
        Co = (((p-1) * (p+2)) / (6 * (g-1))) * (suma2 - (1 / (deno^2)))
        if (Co - (C^2) >= 0) {
            v1 = as.integer((p * (p + 1) * (g - 1)) / 2) ## Numerator DF.
            v21 = as.integer(trunc((v1 + 2) / (Co - (C^2)))) ## Denominator DF.
            F1 = MB * ((1 - C - (v1 / v21)) / v1) ## F approximation.
            ## Significance value associated to the observed F statistic.
            P1 = pf(F1, v1, v21, lower=FALSE)
            cat('\n------------------------------------------------------------\n')
            cat(' MBox F df1 df2 P\n')
            cat('------------------------------------------------------------\n')
            cat(sprintf("%10.4f%11.4f%11.i%14.i%13.4f\n", MB, F1, v1, v21, P1))
            cat('------------------------------------------------------------\n')
            if (P1 >= alpha) {
                cat('Covariance matrices are not significantly different.\n')
            } else {
                cat('Covariance matrices are significantly different.\n')
            }
            return(list(MBox=MB, F=F1, df1=v1, df2=v21, pValue=P1))
        } else {
            v1 = as.integer((p * (p + 1) * (g - 1)) / 2) ## Numerator df.
            v22 = as.integer(trunc((v1 + 2) / ((C^2) - Co))) ## Denominator df.
            b = v22 / (1 - C - (2 / v22))
            F2 = (v22 * MB) / (v1 * (b - MB)) ## F approximation.
            ## Significance value associated to the observed F statistic.
            P2 = pf(F2, v1, v22, lower=FALSE)

            cat('\n------------------------------------------------------------\n')
            cat(' MBox F df1 df2 P\n')
            cat('------------------------------------------------------------\n')
            cat(sprintf('%10.4f%11.4f%11.i%14.i%13.4f\n', MB, F2, v1, v22, P2))
            cat('------------------------------------------------------------\n')

            if (P2 >= alpha) {
                cat('Covariance matrices are not significantly different.\n')
            } else {
                cat('Covariance matrices are significantly different.\n')
            }
            return(list(MBox=MB, F=F2, df1=v1, df2=v22, pValue=P2))
        }
    }
}
