theta2pars.jsz <- function(theta) {
    ## extract individual model parameters from theta vector
    pars <- list(dlamQ=theta[1:cN])
    pars$lamQ=cumsum(pars$dlamQ)
    ## P-innovation covariance matrix
    pars$Sigma <- matrix(0,cN,cN)
    pars$Sigma[lower.tri(pars$Sigma,diag=TRUE)] <- tail(theta,-cN)
    pars$Omega <- pars$Sigma %*% t(pars$Sigma)
    return(pars)
}

pars2theta.jsz <- function(pars) {
    ## convert individual parameters to theta vector
    dlamQ <- c(pars$lamQ[1],diff(pars$lamQ));
    if (length(pars$lamQ)!=cN) stop("lamQ has wrong length")
    Sigma.vec <- pars$Sigma[lower.tri(pars$Sigma,diag=TRUE)]
    return(c(dlamQ,Sigma.vec))
}


theta2pars.jls <- function(theta) {
    pars <- list(rinfQ = theta[1],
                 lamQ = theta[2:(cN+1)])
    pars$gam0 <- matrix(theta[(cN+2):(cN+1+cM)], cM, 1)
    pars$gam1 <- matrix(theta[(cN+1+cM+1):(cN+1+cM+cM*cN)], cM, cN)
    pars$Sigma <- matrix(0, cN, cN);
    pars$Sigma[lower.tri(pars$Sigma,diag=TRUE)] <- tail(theta, cN*(cN+1)/2)
    pars$Omega <- pars$Sigma %*% t(pars$Sigma)
    return(pars)
}

pars2theta.jls <- function(pars) {
    if (length(pars$lamQ)!=cN) stop("lamQ has wrong length")
    Sigma.vec <- pars$Sigma[lower.tri(pars$Sigma,diag=TRUE)]
    theta <- c(pars$rinfQ, pars$lamQ, pars$gam0, as.numeric(pars$gam1), Sigma.vec)
    return(theta)
}

checkPars <- function(pars) {
    ## check parameter restrictions common to all models
    valid <- TRUE
    ## diagonal elements of Sigma positive and bounded away from zero
    if (any(diag(pars$Sigma)<1e-7)) valid <- FALSE
    if (any(diag(pars$Sigma)>1)) valid <- FALSE
    ## Q-eigenvalues not explosive
    if (any(pars$lamQ>0)) valid <- FALSE
    if (any(pars$lamQ< -1)) valid <- FALSE
    ## Q-eigenvalues sorted
    #if (any(pars$dlamQ>0)) valid <- FALSE
    return(valid)
}

obj.jsz <- function(theta) {
    ## objective function for ML estimation
    ## Arguments:
    ##   theta - vector with model parameters
    ## Value: negative of log-likelihood
    ## Globals: W, Y, mats
    pars <- theta2pars.jsz(theta)
    valid <- checkPars(pars)
    if (valid) {
        ## evaluate likelihood function
        res.llk <- jsz.llk(Y, W, K1Q.X=diag(pars$lamQ), Sigma.cP=pars$Omega, mats=mats, dt=1, sigma.e=NA)
        return(sum(res.llk$llk))
    } else {
        ## else return penalty value
        return(1e6)
    }
}

obj.jls <- function(theta) {
    ## objective function for ML estimation of JLS model -- observed risk factors
    ## Arguments:
    ##   theta - vector with model parameters
    ## Value: negative of log-likelihood
    ## Globals: W, Y, M.o, cL, mats

    pars <- theta2pars.jls(theta)
    valid <- checkPars(pars)
    if (pars$rinfQ<0) valid <- FALSE
    if (pars$rinfQ*1200>20) valid <- FALSE
    if (valid) {
        res.llk <- jls.llk(Y, M.o, W, cL, rinfQ=pars$rinfQ, lamQ=pars$lamQ, gam0=pars$gam0, gam1=pars$gam1, Sigma=pars$Sigma, mats=mats, dt=1)
        obj <- sum(res.llk$llk)
    } else {
        obj <- 1e6
    }
}

getStartingValuesJSZ <- function(Sigma) {
    ## obtain starting values for lamQ for MLE
    ##
    ## Arguments: Sigma
    ## Value: list with starting values
    ## Globals: Y, W, mats
    if (missing(Sigma))
        error("Sigma needs to be provided")
    N <- nrow(W) # number of factors
    nSeeds <- 100;  # how many random seeds to try
    best.llk <- Inf
    Omega <- Sigma %*% t(Sigma)
    for (i in 1:nSeeds) {
        lamQ <- -sort(abs(.01*rnorm(N)))
        res.llk <- jsz.llk(Y, W, K1Q.X=diag(lamQ), Sigma.cP=Omega, mats=mats, dt=1)
        llk <- sum(res.llk$llk)
        if (llk < best.llk) {
            cat('Improved seed llk to ', llk, '\n')
            best.llk <- llk
            best.lamQ <- lamQ
        }
    }
    return(list(lamQ=best.lamQ, Sigma=Sigma))
}

getStartingValuesJLS <- function(Sigma) {
    ## get good starting values for JLS model estimation using random seeds
    ## Arguments: Sigma from VAR
    ## Value: list with starting values
    ## Globals: cM, cN, WN, M.o, n.per

    ## starting values for gam0 and gam1: regress macro on cN yield factors
    gam0 <- matrix(NA, cM, 1)
    gam1 <- matrix(NA, cM, cN)
    PN.o <- Y %*% t(WN)
    xdat <- PN.o
    for (i in 1:cM) {
        ydat <- M.o[,i]
        res <- lm(ydat ~ xdat)
        gam0[i] <- res$coef[1]
        gam1[i,] <- res$coef[2:(cN+1)]
    }

    ## random starting values for lamQ and rinfQ
    n.seeds <- 100
    best.llk <- Inf
    pars <- list(gam0 = gam0, gam1 = gam1, Sigma = Sigma)
    for (i in 1:n.seeds) {
        pars$lamQ <- -sort(abs(.1*rnorm(cN)))
        pars$rinfQ <- rnorm(1, mean=8/100/n.per, sd=4/100/n.per)
        ## pars$gam0 = gam0 + matrix(rnorm(cM, mean=gam0, sd=abs(gam0)/2), cM, 1)
        ## pars$gam1 = gam1 + matrix(rnorm(cM*cN, mean=gam1, sd=abs(gam1)/2), cM, cN)
        theta <- pars2theta.jls(pars)
        llk <- obj.jls(theta)
        if (llk < best.llk) {
            cat('Improved seed llk to ', llk, '\n')
            best.llk <- llk
            best.pars <- pars
        }
    }
    return(best.pars)
}

getOptim <- function (theta, obj, ...)
{
    obj <- match.fun(obj)
    cat("Starting optimization...\n")
    cat("Function value at starting point = ", sprintf("%10.4f",
        obj(theta, ...)), "\n")
    i <- 1
    improvement <- Inf
    prev.llk <- 0
    while (improvement > 0.1) {
        myparscale <- 10^round(log10(abs(theta)))
        res <- optim(theta, obj, gr = NULL, ..., control = list(parscale = myparscale))
        improvement <- abs(res$value - prev.llk)
        prev.llk <- res$value
        theta <- res$par
        cat("iteration ", i, ", likelihood = ", sprintf("%10.4f",
            res$value), "\n")
        i <- i + 1
    }
    cat("improvement = ", improvement, " -- proceed to final step\n")
    res <- optim(theta, obj, gr = NULL, ..., control = list(trace = 0,
        maxit = 50000, parscale = myparscale))
    cat("final Nelder-Mead step, likelihood = ", sprintf("%10.4f",
        res$value), "\n")
    cat("Convergence:", res$convergence, "\n")
    print(res$message)
    return(res$par)
}
