load_model <- function(filename, flag.zlb, flag.macro, rmin=0, flag.fitted=TRUE) {
    require(jsz)
    require(jls)
    ## loads model parameters and filters factors
    ## Value: list with all parameters and information about the model

    cat("# loading", ifelse(flag.zlb, "shadow-rate", "affine"), "model:", filename, "\n")

    ## load yield data (if not loaded yet)
    if (!exists('Y'))
        loadYieldData()
    path <- ifelse('estimates' %in% dir(), "estimates/", "../estimates/")
    res.file <- paste0(path, filename)
    model <- loadPars(res.file, flag.macro)
    if (flag.macro) {
        cat("macro-finance model - ")
        ## load macro data (if not loaded yet)
        if (!exists('M.o'))
            loadMacroData()
        cat("number of yield factors: L = ", model$cL, "\n")
        ## Kalman filter
        if (flag.zlb) {
            ## shadow-rate macro-finance model
            cat('filtering ZLB risk factors...\n')
            kalman <- jls.kalman.zlb(Y, M.o, model$cL, K0P=model$K0P, K1P=model$K1P, Sigma=model$Sigma, rho0=model$rho0, rho1=model$rho1, K0Q=model$K0Q, K1Q=model$K1Q, mats, rmin, dt=1, sigma.e=model$sigma.e)
        } else {
            ## affine macro-finance model
            kalman <- jls.kalman(Y, M.o, model$W, model$cL, rinfQ=model$rinfQ, lamQ=model$lamQ, gam0=model$gam0, gam1=model$gam1, K0P=model$K0P, K1P=model$K1P, Sigma=model$Sigma, mats, dt=1, sigma.e=model$sigma.e)
        }
        model$cP <- kalman$x.tt
        model$llk <- -sum(kalman$llk)
    } else {
        cat("yields-only model - ")
        cat("number of yield factors: N = ", model$cN, "\n")
        ## Kalman filter
        if (flag.zlb) {
            ## shadow-rate yields-only model
            cat('filtering ZLB risk factors...\n')
            kalman <- jsz.kalman.zlb(Y, model$mu, model$Phi, model$Sigma, model$muQ, model$PhiQ, model$rho0, model$rho1, model$sigma.e, mats, rmin)
        } else {
            ## affine yields-only model
            kalman <- jsz.kalman(Y, model$W, model$mu, model$Phi, model$Sigma, model$kinfQ, model$lamQ, model$sigma.e, mats)
        }
        model$cP <- kalman$x.tt
        model$llk <- kalman$llk.sum
    }
    ## shadow/short rates, shadow/fitted yields
    if (flag.fitted) {
        T <- nrow(Y)
        if (flag.zlb) {
            model$s <- rep(model$rho0.cP, T) + model$cP %*% model$rho1.cP
            model$r <- pmax(rmin, model$s)
            model$Y.shadow <- rep(1,T) %*% model$A + model$cP %*% model$B
            ## fitted yields
            model$Y.hat <- getY(model$cP, model$rho0.cP, model$rho1.cP, model$muQ, model$PhiQ, model$Sigma, rmin)
            ## risk-neutral rates
            model$Y.rn <- getY(model$cP, model$rho0.cP, model$rho1.cP, model$mu, model$Phi, model$Sigma, rmin)
        } else {
            model$r <- rep(model$rho0.cP, T) + model$cP %*% model$rho1.cP
            model$Y.hat <-rep(1,T) %*% model$A + model$cP %*% model$B
            loads.rn <- gaussian.loadings(mats, model$mu, model$Phi-diag(model$cN), model$Omega, model$rho0.cP, model$rho1.cP)
            model$Arn <- loads.rn$A; model$Brn <- loads.rn$B
            model$Y.rn <-  rep(1,T)%*%model$Arn + model$cP%*%model$Brn
        }
        ## term premia
        model$Y.tp <- model$Y.hat - model$Y.rn
    }
    model$res.file <- res.file
    model$flag.zlb <- flag.zlb
    model$flag.macro <- flag.macro
    if (flag.zlb)
        model$rmin <- rmin
    return(model)
}

loadPars <- function(file.name, flag.macro=FALSE) {
    ## load parameters from file and construct additional parameters
    ## Arguments:
    ##   file.data -- RData file with estimation results
    ##   flag.macro -- flag to indicate whether results are for macro-finance model
    ## Value:  list with model parameters
    ## Globals: uses mats
    require(jsz)

    load.list <- load(file.name)
    pars$W <- W
    if (ncol(W)!=ncol(Y))
        stop("W loaded from file does not have the same number of yield maturities as yield data in Y")
    if (flag.macro) {
        ## macro-finance model
        if (!("gam0" %in% names(pars)))
            stop("flag.macro is TRUE but no JLS parameters saved in file")
        pars$cL <- cL
        pars$cN <- cN
        pars$cM <- cM
        loads <- jls.loadings(W, pars$rinfQ, pars$lamQ, pars$gam0, pars$gam1, pars$Omega, mats, dt=1)
        pars$A <- loads$A; pars$B <- loads$B
        pars$muQ <- as.numeric(loads$K0Q)
        pars$PhiQ <- loads$K1Q + diag(cN)
        pars$mu <- pars$K0P
        pars$Phi <- pars$K1P + diag(cN)
        pars$K0Q <- as.numeric(loads$K0Q);
        pars$K1Q <- loads$K1Q
    } else {
        ## yields-only model
        if (!("kinfQ" %in% names(pars)))
            stop("flag.macro is FALSE but no JSZ parameters saved in file")
        if ("cN" %in% load.list)
            pars$cN <- cN
        if ("N" %in% load.list)
            pars$cN <- N
        if (pars$cN != length(pars$lamQ))
            stop("Saved value of cN not consistent with parameter estimates")
        loads <- jsz.loadings(W, diag(pars$lamQ), pars$kinfQ, pars$Omega, mats, dt=1)
        pars$A <- loads$AcP; pars$B <- loads$BcP
        pars$muQ <- as.numeric(loads$K0Q.cP)
        pars$PhiQ <- loads$K1Q.cP + diag(pars$cN)
        pars$K0Q <- as.numeric(loads$K0Q.cP);
        pars$K1Q <- loads$K1Q.cP
    }
    pars$rho0.cP <- as.numeric(loads$rho0)
    pars$rho1.cP <- as.numeric(loads$rho1)
    return(pars)
}

assignNames <- function(models) {
    for (i in seq_along(models)) {
        models[[i]]$name <- names(models)[i]
    }
    models
}

isPosDef <- function(M) {
    if ( all(M == t(M) ) ) {  # first test symmetricity
        if (  all(eigen(M)$values > 0) ) {TRUE}
        else {FALSE} } #
    else {FALSE}  # not symmetric
}

Mode <- function(x) {
  ux <- unique(x)
  ux[which.max(tabulate(match(x, ux)))]
}

irf.var1 <- function(Phi, max.lag = 500, g=1, h=1) {
    ## calculate impulse response function for a VAR(1)
    ## for the g'th variable in response to shocks to the h'th variable

    if (length(Phi)>1) {
        k <- dim(Phi)[1]
        Psi <- array(0, c(max.lag,k,k))
        Psi[1,,] <- Phi
        for (i in 2:max.lag)
            Psi[i,,] <- Phi %*% Psi[i-1,,]
        irf.var1 <- Psi[,g,h]
    } else {
        Psi <- numeric(max.lag)
        for (i in 1:max.lag)
            Psi[i] <- Phi^i
        irf.var1 <- Psi
    }
}

convertDate <- function(date)
    ## convert numeric date YYYYMMDD to Date object
    return(as.Date(as.character(date), format="%Y%m%d"))

getEndDate <- function(flag.subsmpl)
    return(ifelse(flag.subsmpl, 20080000, 20150000))

getDataPath <- function(file)
    paste(
        ifelse('data' %in% dir(), "data/", "../data/"),
        file, sep="")

loadYieldData <- function(end.date, flag.subsmpl=FALSE) {
    ## load yield data
    ## Globals: creates n.per, mats, Y, dates
    n.per <<- 12

    cat("Loading yield data...\n")

    ## GSW data
    load(getDataPath("gsw_data_monthly.RData"))  # Y, dates, mats
    mat.sel <- c(1,2,3,4,5,7,9,12) # original results
    ##mat.sel <- c(3:12) # for new estimation 1/27/2014
    Y <- Y[,mat.sel]/100/n.per
    mats <- mats[mat.sel]*n.per  ## in months

    start.date <- 19850101
    if (missing(end.date))
        end.date <- getEndDate(flag.subsmpl)
    sel.sample <- dates>=start.date & dates<=end.date
    Y <- Y[sel.sample,]
    dates <- dates[sel.sample]
    cat("Start date:", min(dates), "\n")
    cat("End date:", max(dates), "\n")

    ## only now make global
    ## (otherwise changes above are not made to local copy
    dates <<- dates
    mats <<- mats
    Y <<- Y
}

loadMacroData <- function() {
    ## load macro data
    ## Globals: creates M.o; uses Y
    cat("Loading macro data...\n")

    macro.data <- read.csv(getDataPath("macro_monthly_cpi_ugap.csv"))

    end.date <- max(dates) ## from yield data

    macro.data <- macro.data[macro.data$DATE<=end.date,]
    cat("Start date:", min(macro.data$DATE), "\n")
    cat("End date:", max(macro.data$DATE), "\n")

    if (nrow(macro.data)!=dim(Y)[1]) stop("yield and macro data do not have the same length")
##    M.o <<- as.matrix(macro.data[,2:3])  # construct macro factors

    ## CHANGE
    M.o <<- as.matrix(macro.data[,2:3])/1200  # construct macro factors
}

init.jsz <- function(flag.subsmpl=TRUE, cN=3) {
    ## initialize model settings
    ## load data
    ## construct factors
    ##
    ## Globals:
    ##  creates W

    loadYieldData(flag.subsmpl=flag.subsmpl)
    eig <- eigen(cov(Y))
    W <<- t(eig$vectors[,1:cN])
}

init.jls <- function(flag.subsmpl=TRUE, cL=1) {
    ## initialize model settings
    ## load data
    ## construct factors
    ##
    ## Globals:
    ##  creates cM, W, WL, WN, M.o
    ##  uses mats

    loadYieldData(flag.subsmpl=flag.subsmpl)
    loadMacroData()

    if (dim(M.o)[2]!=2)
        stop("number of macro factors is not 2")

    cM <<- 2
    cN <<- cL+cM

    eig <- eigen(cov(Y))
    W <<- t(eig$vectors)  ## rows contain loadings, first row is first PC
    WL <<- matrix(W[1:cL,], cL, length(mats))
    WN <<- matrix(W[1:cN,], cN, length(mats))
}

getY <- function(cP, rho0, rho1, mu, Phi, Sigma, rmin=0, flag.zlb=TRUE, method = c("CGF2", "Krippner", "MC", "CGF1"), M=500) {
    method <- match.arg(method)
    ## shortcut to get quick results:
    method <-  "Krippner"
    if (method == "CGF2")
        ## 2nd order CGF approximation of Priebsch
        getYfn <- make.getY.CGF(rho0, rho1, mu, Phi, Sigma, rmin=rmin, flag.zlb=flag.zlb, second.order = TRUE)
    else if (method == "CGF1")
        ## 1st order approximation - ignored convexity
        getYfn <- make.getY.CGF(rho0, rho1, mu, Phi, Sigma, rmin=rmin, flag.zlb=flag.zlb, second.order = FALSE)
    else if (method == "Krippner")
        ## Krippner
        getYfn <- make.getY.Krippner(rho0, rho1, mu, Phi, Sigma, rmin=rmin, flag.zlb=flag.zlb)
    else if (method == "MC")
        ## Monte Carlo - quite inaccurate
        getYfn <- function(Xt) getY.MC(Xt, rho0, rho1, mu, Phi, Sigma, rmin=rmin, flag.zlb=flag.zlb)

    if (is.matrix(cP) && dim(cP)[2]==length(rho1)) {
        ## matrix -- several/all dates
        T <- nrow(cP)
        Yhat <- matrix(NA, T, length(mats))
        for (t in 1:T)
            Yhat[t,] <- getYfn(cP[t,])
    } else {
        ## vector -- only one date
        Yhat <- getYfn(cP)
    }
    Yhat
}

## ZLB bond prices - Monte Carlo
getY.MC <- function(x0, gamma.0, gamma.1, muQ, PhiQ, Sigma, rmin=0, M=500, flag.zlb=TRUE, fixed.seed=TRUE) {
    ## calculate yields for shadow-rate model
    ## Arguments:
    ##  x0 - risk factors at time t0
    ##  gamma.0, gamma.1 - coefficients in shadow-rate equation
    ##  muQ, PhiQ - risk-neutral dynamics
    ##  Sigma - Cholesky decomposition of shock cov matrix
    ##  rmin - lower bound on nominal interest rates (as decimal number per period, e.g. 15/120000)
    ##  M - number of Monte Carlo replications
    ##  flag.zlb - TRUE: shadow-rate model, FALSE: affine model
    ##  fixed.seed - TRUE: always use same seed for random number generation

    x0 <- as.numeric(x0)
    gamma.1 <- as.numeric(gamma.1)
    if (length(x0)!=length(gamma.1)) {
        print(x0)
        print(gamma.1)
        stop("x0 and gamma.1 not consistent")
    }
    muQ <- as.numeric(muQ)

    ## max horizon in simulation -- longest maturity
    H <- max(mats)

    ## number of risk factors
    N <- length(gamma.1)

    ## sampled paths of the short rate
    r.sim <- matrix(NA, M, H)
    if (flag.zlb) {
        r.sim[,1] <- max(gamma.0 + crossprod(gamma.1, x0), rmin)   ## ZLB
    } else {
        r.sim[,1] <- gamma.0 + crossprod(gamma.1, x0)   ## affine
    }

    mu.mat <- matrix(rep(muQ, M), N, M)
    x.mat <- matrix(rep(x0, M), N, M)
    gam0.vec <- rep(gamma.0, M);

    shocks <- matrix(NA, N, M);
    ind.even <- seq(2,M,2)
    ind.odd <- seq(1,M,2)

    if (fixed.seed)
        set.seed(10)

    for (h in 2:H) {
        tmp <- matrix(rnorm(M*N/2), N, M/2)
        shocks[, ind.even] <- tmp;
        shocks[, ind.odd] <- -tmp;   # antithetic sampling
        x.mat <- mu.mat + PhiQ %*% x.mat + Sigma %*% shocks
        if (flag.zlb) {
            r.sim[,h] <- pmax( gam0.vec + gamma.1 %*% x.mat, rmin)   ## ZLB
        } else {
            r.sim[,h] <- gam0.vec + gamma.1 %*% x.mat           ## affine
        }
    }

    ## return bond yields
    ## 1. calculate numerical bond prices
    Pn.mc <- numeric(length(mats))
    for (tau in mats)
        Pn.mc[which(tau==mats)] <- mean( exp(-rowSums(r.sim[,1:tau])))
    ## 2. calculate yields
    Y.mc <- -log(Pn.mc)/mats
    ## raise error if NA (e.g., logs of negative values)
    if (any(is.na(Y.mc))) {
        stop("error in Yzlb -- some yields are NA")
    } else {
        return(Y.mc)
    }
}

make.getY.CGF <- function(rho0, rho1, mu, Phi, Sigma, H=max(mats), rmin=0, flag.zlb=TRUE, second.order=TRUE) {
    ## calculate yields for shadow-rate model using cumulant-generating-function approach
    ## this is the function factory -- carries out all calculations not depending on Xt
    ## Globals: mats

    ##require(mnormt)
    require(pbivnorm)

    ## do not depend on Xt:
    Omega <- Sigma %*% t(Sigma)
    sig2.af <- getVarAffine(Phi, Omega, rho1, H-1)         # Var_t(r(t+h))  h=1,..,H-1
    sig.af <- sqrt(sig2.af)

    ## calculate 2nd moments of affine short rates
    ## Cov.af[g, h] = Cov_t(r(t+g), r(t+h))
    ## 1. build of matrix with summands
    A <- matrix(NA, H-1, H-1)  # A_ij = rho1' phi^{i-1} Omega phi^{j-1}' rho1
    Bi1 <- Omega # phi^{i-1} Omega
    A[1, 1] <- t(rho1) %*% Bi1 %*% rho1
    for (i in 2:(H-1)) {
        Bi1 <- Phi %*% Bi1
        A[i, 1] <- t(rho1) %*% Bi1 %*% rho1
        Bij <- Bi1  # phi^{i-1} Omega phi^{j-1}'
        for (j in 2:i) {
            Bij <- Bij %*% t(Phi)
            A[i, j] <- t(rho1) %*% Bij %*% rho1
        }
    }
    ## fill upper triangular of A
    A[upper.tri(A)] <- t(A)[upper.tri(A)]
    ## 2 sum up appropriately
    Cov.af <- matrix(NA, H-1, H-1) ##
    for (g in 1:(H-1))
        for (h in 1:g) {
            i <- 1:min(g, h)
            select <- cbind(g-i+1, h-i+1)
            Cov.af[g, h] <- sum(A[select])
        }
    Cov.af[upper.tri(Cov.af)] <- t(Cov.af)[upper.tri(Cov.af)]

    stopifnot(isTRUE(all.equal(sig2.af, diag(Cov.af), check.attr=FALSE)))

    ## determine remaining necessary parameters and variables
    Cor.af <- cov2cor(Cov.af)
    sqrt.2pi <- sqrt(2*pi)
    Err.black <- matrix(0, H-1, H-1)
    ## indexing lower-triangular elements of MxM matrix Err.black (without diagonal)
    M <- H-1
    i <- 1:(M*(M-1)/2)
    ii <- M*(M-1)/2-i
    K <- floor((sqrt(8*ii+1)-1)/2)
    h <- M-K-1          ## column index
    jj <- ii - K*(K+1)/2
    g <- M - jj         ## row index
    chivec <- Cor.af[cbind(g, h)]
    sqrt.chi <- sqrt(1-chivec^2)
    Covvec <- Cov.af[cbind(g, h)]

    lt <- as.integer(length(i))
    prob <- double(lt)
    mypbivnorm <- function(x, y, rho)
        .Fortran("PBIVNORM", prob, c(0, 0), x, y,
                 c(0, 0), rho, lt, PACKAGE = "pbivnorm")[[1]]

    function(Xt) {
        Xt <- as.numeric(Xt)
        mu.af <- getMeanAffine(Xt, mu, Phi, rho0, rho1, H-1)   # H x 1
        s_t <- rho0 + crossprod(rho1, Xt)
        if (flag.zlb) {
            ## shadow-rate model
            ## Err.black[g, h] = E_t(r(t+g), r(t+h))   g=1,...,H-1  h=1,...,H-1
            s <- mu.af/sig.af
            ps <- pnorm(s)
            ds <- dnorm(s)
            r_t <- max(s_t, 0)
            mu.black <- mu.af*ps + sig.af*ds  # H-1 x 1
            ## ER[g] = E_t(R_{t,t+g-1})    g=1,...,H
            ER <- cumsum(c(r_t, mu.black))
            if (second.order) {
                ## 2nd-order approx -- need 2nd moments
                ## vectorized calls to dnorm and pnorm
                p1 <- pnorm((s[g]-chivec*s[h])/sqrt.chi)
                p2 <- pnorm((s[h]-chivec*s[g])/sqrt.chi)
                d3 <- dnorm(sqrt(s[g]^2-2*chivec*s[g]*s[h]+s[h]^2)/sqrt.chi)
                pbiv <- mypbivnorm(-s[g], -s[h], chivec)
                Err.black[] <- 0
                ## cross-moments of censored normal random variables
                Err.black[lower.tri(Err.black)] <- (mu.af[g] * mu.af[h] + Covvec)*(ps[g] + ps[h] - 1 + pbiv) + sig.af[h]*mu.af[g]*ds[h]*p1[i] + sig.af[g]*mu.af[h]*ds[g]*p2[i] + sig.af[g]*sig.af[h]*sqrt.chi/sqrt.2pi*d3[i]
                Err.black[upper.tri(Err.black)] <- t(Err.black)[upper.tri(Err.black)]
                diag(Err.black) <- (mu.af^2 + sig2.af)*ps + sig.af*mu.af*ds
                ## Err[i,j] = E_t(r_{t+i-1}, r_{t+j-1})   i = 1,...,H  j = 1,...,H
                Err <- rbind(c(r_t^2, r_t*mu.black),
                             cbind(r_t*mu.black, Err.black))
            }
        } else {
            ## affine model
            ER <- cumsum(c(s_t, mu.af))
            if (second.order)
                Err <- rbind(c(s_t^2, s_t*mu.af),
                             cbind(s_t*mu.af, Cov.af + outer(mu.af, mu.af)))

        }

        ## calculate first- and second-order approximations
        if (second.order) {
            ## E_t(R^2)
            ER2 <- cumsum(vapply(1:H, function(h) {
                                     rval <- Err[h,h]
                                     if (h>1) {
                                         rval <- rval + 2*sum(Err[1:(h-1),h])
                                     }
                                     rval }, 0))
            ##ER2 <- vapply(1:H, function(i) sum(Err[1:i, 1:i]), 0)
            yields <- rmin + 1/(1:H)*(ER - 0.5*(ER2 - ER^2))
        } else {
            yields <- rmin + 1/(1:H)*ER
        }
        yields[mats]
    }
}

make.getY.Krippner <- function(rho0, rho1, mu, Phi, Sigma, rmin=0, flag.zlb=TRUE) {
    ## calculate yields for shadow-rate model using Krippner approx
    ## this is the function factory -- carries out all calculations not depending on Xt
    ## Globals: mats
    H <- max(mats)

    ## affine forward rates: f.af[h] = Af[h] + Bf[,h] %*% Xt
    ## affine expectations:  mu.af[h]  = Abar[h] + Bf[,h] %*% Xt
    ## variance of affine short rate: sig[h] = Var_t(s(t+h))
    ## affine bond price: P.af[h] = A[h] + B[,h] %*% Xt
    ## affine yield: y.af[] = -A[h]/h - B[,h]/h %*% Xt
    ## shadow-rate expectations: mu.sr[h] = rmin + sig[h]*g((mu.af[h]-rmin)/sig[h])
    ## Krippner forward rates:   f.sr[h] = rmin + sig[h]*g( (f.af[h]-rmin)/sig[h])
    g <- function(z)
        return(z*pnorm(z) + dnorm(z))
    N <- length(mu) ## number of risk factors
    Omega <- Sigma %*% t(Sigma)
    Abar <- numeric(H)
    Af <- numeric(H); Bf <- matrix(NA, N, H) # forward rate loadings
    A <- numeric(H); B <- matrix(NA, N, H)  # bond price loadings

    f.af <- numeric(H) # affine forward rates
    y.af <- numeric(H) # affine yields
    mu.af <- numeric(H)   # affine expectations
    f.sr <- numeric(H) # Black forward rates
    y.sr <- numeric(H) # Black yields
    mu.sr <- numeric(H) # Black expectations
    sig <- numeric(H) # conditional volatility of shadow rate/affine short rate

    A[1] <- -rho0
    B[,1] <- -rho1
    Af[1] <- rho0 - crossprod(B[,1], mu) - .5*t(B[,1])%*%Omega%*%B[,1]
    Abar[1] <- rho0 - crossprod(B[,1], mu)
    Bf[,1] <- t(Phi) %*% rho1
    Omega <- Sigma %*% t(Sigma)
    VarX <- Omega  ## one-step ahead forecast error covariance
    sig[1] <- sqrt(t(rho1) %*% VarX %*% rho1)
    Phi.hm1 <- Phi
    for (h in 2:H) {
        ## affine yields
        A[h] <- A[h-1] + crossprod(mu, B[,h-1]) + .5*t(B[,h-1])%*%Omega%*%B[,h-1] - rho0
        B[,h] <- t(Phi) %*% B[,h-1] - rho1
        ## see also gaussian.loadings in JSZ package
        ## affine forward rates and expectations
        Abar[h] <- rho0 - crossprod(B[,h], mu)
        Af[h] <- Abar[h] - .5*t(B[,h])%*%Omega%*%B[,h]
        Bf[,h] <- t(Phi) %*% Bf[,h-1]
        ## conditional variance
        VarX <- VarX + Phi.hm1 %*% Omega %*% t(Phi.hm1)
        Phi.hm1 <- Phi %*% Phi.hm1    ## Phi^(h-1)
        sig[h] <- sqrt(t(rho1) %*% VarX %*% rho1)
    }

    function(Xt) {
        mu.af <- Abar + t(Bf) %*% Xt
        f.af <- Af + t(Bf) %*% Xt
        r.af <- rho0 + crossprod(rho1, Xt)
        y.af <- cumsum(c(r.af, f.af[1:(H-1)]))/(1:H)
        f.sr <- rmin + sig*g((f.af-rmin)/sig)
        mu.sr <- rmin + sig*g((mu.af-rmin)/sig)
        r.sr <- max(rmin, r.af)
        y.sr <- cumsum(c(r.sr, f.sr[1:(H-1)]))/(1:H)
        if (flag.zlb) {
            y.sr[mats]
        } else {
            y.af[mats]
        }
    }
}

jsz.kalman.zlb <- function(Y, mu, Phi, Sigma, muQ, PhiQ, rho0, rho1, sigma.e, mats, rmin, smoothing=FALSE) {
    require(numDeriv)
    ## likelihood function for ZLB affine model
    ## evaluation using Extended Kalman filter
    ## Arguments:
    ##   Y - data
    ##   pars - list of model parameters (muQ, PhiQ, Omega, Phi, mu, sigma.e)
    ##   mats - yield maturities
    ##   smoothing - flag indicating whether to calculate smoothed yields
    ## Value:
    ##   list
    ##     llk.sum - log-likelihood
    ##     x.tt - filtered risk factors
    ##     x.tT - smoothed risk factors

    T <- nrow(Y)
    N <- length(mu)
    J <- ncol(Y)

    ## parameters state equation
    ## X_t = mu + Phi * X_t-1 + v_t     Q = Cov(v_t)
    Omega <- Sigma %*% t(Sigma)
    Q <- Omega

    ## parameters measurement equation
    ## parameters measurement equation
    ## Y_t = g(X_t) + e_t          R = Cov(e_t)
    R <- diag(J)*sigma.e^2   # J x J

    x.tt <- matrix(NA, T, N)
    x.ttm1 <- matrix(NA, T, N)
    P.tt <- array(NA, c(T, N, N))
    P.ttm1 <- array(NA, c(T, N, N))

    ## starting values for x and P
    x.ttm1[1,] <- solve(diag(N)-Phi) %*% mu
    P.ttm1[1,,] <- matrix( solve(diag(N^2) - kronecker(Phi, Phi))%*%as.numeric(Q), N, N);

    ## shortcut to get quicker results
    getY <- make.getY.Krippner(rho0, rho1, muQ, PhiQ, Sigma, rmin=rmin)
    ## getY <- make.getY.CGF(rho0, rho1, muQ, PhiQ, Sigma, rmin=rmin, second.order = TRUE)

    llk.sum <- 0
    for (t in 1:T) {
        ## calculate numerical derivative
        H <- jacobian(getY, x.ttm1[t,], method="simple")
        tH <- t(H);
        if (any(is.na(H))) {
            cat("Jacobian contains NaN elements")
            browser()
        }
        V <- H %*% P.ttm1[t,,] %*% tH + R  # MSFE, N x N
        V.inv <- solve(V)
        K <- Phi %*% P.ttm1[t,,] %*% tH %*% V.inv  # Kalman gain
        eta <- Y[t,] - getY(x.ttm1[t,])
        llk.sum <- llk.sum -.5*log(det(V)) -.5 * t(eta) %*% V.inv %*% eta

        x.tt[t,] <- x.ttm1[t,] + P.ttm1[t,,] %*% tH %*% V.inv %*% eta
        P.tt[t,,] <-  P.ttm1[t,,] - P.ttm1[t,,] %*% tH %*% V.inv %*% H %*% P.ttm1[t,,] #P_t|t

        if (t<T) {
            ## update next periods values
            x.ttm1[t+1,] <- mu + Phi %*% x.ttm1[t,] + K %*% eta  # x_t+1|t
            P.ttm1[t+1,,] <- Phi %*% P.tt[t,,] %*% t(Phi) + Q  # P_t+1|t
        }
    }
    if (smoothing) {
        ## smoothing
        x.tT <- matrix(NA, T, N)
        x.tT[T,] <- x.tt[T,]
        for (t in (T-1):1) {
            J.t <- P.tt[t,,] %*% t(Phi) %*% solve(P.ttm1[t+1,,])
            x.tT[t,] <- x.tt[t,] + J.t %*% (x.tT[t+1,] - x.ttm1[t+1,])
        }
        return(list(llk.sum=llk.sum, x.tt=x.tt, x.tT=x.tT))
    } else {
        return(list(llk.sum=llk.sum, x.tt=x.tt))
    }
}

jls.kalman.zlb <- function(yields.o, M.o, cL, K0P, K1P, Sigma, rho0, rho1, K0Q, K1Q, mats, rmin, dt, sigma.e, sigma.m, smoothing=FALSE) {
    require(numDeriv)
    ## setup
    T <- nrow(yields.o)
    J <- ncol(yields.o)
    cM <- ncol(M.o)
    cN <- cL + cM
    Omega <- Sigma %*% t(Sigma)
    Phi <- K1P + diag(cN)
    mu <- K0P
    PhiQ <- K1Q + diag(cN)
    muQ <- K0Q

    ## parameters state equation
    Q <- Omega

    ## parameters measurement equation
    ## 1. J equations for yields         yt^o = f(Zt) + e_t
    Omega.e <- diag(J)*sigma.e^2
    ## 2. cM equations for macro factors Mt^o = Mt + eta_t
    if (missing(sigma.m)) {
        Omega.M <- matrix(0, cM, cM) # observed without error
    } else {
        if (length(sigma.m)!=cM) stop("length of sigma.m is wrong")
        Omega.M <- diag(sigma.m)^2
    }

    R <- rbind( cbind(Omega.e, matrix(0,J,cM)), cbind(matrix(0,cM,J), Omega.M) ) # (J+cM) x (J+cM)

    T <- dim(yields.o)[1]
    x.tt <- matrix(NA, T, cN)
    x.ttm1 <- matrix(NA, T, cN)
    P.tt <- array(NA, c(T, cN, cN))
    P.ttm1 <- array(NA, c(T, cN, cN))

    ## starting values for x and P
    x.ttm1[1,] <- solve(diag(cN) - Phi) %*% mu
    P.ttm1[1,,] <- matrix( solve(diag(cN^2) - kronecker(Phi, Phi))%*%as.numeric(Q), cN, cN);

    ## shortcut to get quicker results
    getY <- make.getY.Krippner(rho0, rho1, muQ, PhiQ, Sigma, rmin=rmin)
    ## getY <- make.getY.CGF(rho0, rho1, muQ, PhiQ, Sigma, rmin=rmin, second.order = TRUE)

    llk <- numeric(T)
    for (t in 1:T) {
        ## derivative of yields w.r.t risk factors -- numerical derivative
        H.Y <- jacobian(getY, x.ttm1[t,], method="simple")
        if (any(is.na(H.Y))) {
            stop("Jacobian contains NaN elements")
        }
        ## derivative of macro factors w.r.t risk factors -- ones and zeros
        H.M <- cbind(diag(cM),matrix(0, cM, cL))
        H <- rbind(H.Y, H.M)

        V <- H %*% P.ttm1[t,,] %*% t(H) + R  # MSFE, cN x cN
        V.inv <- solve(V)
        K <- Phi %*% P.ttm1[t,,] %*% t(H) %*% V.inv  # Kalman gain
        eta.Y <- yields.o[t,] - getY(x.ttm1[t,])
        eta.M <- M.o[t,] - x.ttm1[t,1:cM]
        eta <- c(eta.Y, eta.M)
        if (det(V)<=0) stop("V not positive definite in Kalman filter")
        llk[t] <- -.5*log(det(V)) -.5 * t(eta) %*% V.inv %*% eta

        x.tt[t,] <- x.ttm1[t,] + P.ttm1[t,,] %*% t(H) %*% V.inv %*% eta
        P.tt[t,,] <-  P.ttm1[t,,] - P.ttm1[t,,] %*% t(H) %*% V.inv %*% H %*% P.ttm1[t,,] #P_t|t

        if (t<T) {
            ## update next periods values
            x.ttm1[t+1,] <- mu + Phi %*% x.ttm1[t,] + K %*% eta  # x_t+1|t
            ## should be equal to mu + Phi %*% x.tt[t]
            P.ttm1[t+1,,] <- Phi %*% P.tt[t,,] %*% t(Phi) + Q  # P_t+1|t
        }
    }

    if (smoothing) {
        ## smoothing
        x.tT <- matrix(NA, T, cN)
        x.tT[T,] <- x.tt[T,]
        for (t in (T-1):1) {
            J.t <- P.tt[t,,] %*% t(Phi) %*% solve(P.ttm1[t+1,,])
            x.tT[t,] <- x.tt[t,] + J.t %*% (x.tT[t+1,] - x.ttm1[t+1,])
        }
        return(list(llk=-llk, x.tt=x.tt, x.tT=x.tT))
    } else {
        return(list(llk=-llk, x.tt=x.tt))
    }
}

getModelName <- function(model) {
    name <- model$name
    if (nchar(name)>3) {
        return(name)
    } else {
        return(paste(substr(name, 1, 2), "(", substr(name, 3, 3), ")", sep=""))
    }
}

printFit <- function(models, to.latex=FALSE) {
    ## print cross-sectional fit
    ## Table 1
    ## Globals: uses zlb.ind, mats
    getFitTbl <- function(subsmpl=FALSE) {
        if (subsmpl) {
            ind <- zlb.ind
        } else {
            ind <- 1:nrow(Y)
        }
        out.matrix <- matrix(NA, length(models), 1+length(mats))
        colnames(out.matrix) <- c("Total", as.character(mats))
        rownames(out.matrix) <- sapply(models, getModelName)
        row <- 1
        cat("Yields min: ", min(Y)*1200, "\n")
        for (model in models) {
            ## if ('Y' %in% names(model))
            ##     Y <- model$Y
            ## debug
            cat("Model", model$name, "\n")
            cat("Fitted yields min: ", min(model$Y.hat)*1200, "\n")
            RMSE <- sqrt(mean((Y[ind,]-model$Y.hat[ind,])^2))*n.per*10000
            RMSEs <- sqrt(colMeans((Y[ind,]-model$Y.hat[ind,])^2))*n.per*10000
            out.matrix[row,] <- c(RMSE, RMSEs)
            row <- row+1
        }
        return(out.matrix)
    }

    cat("Full sample\n")
    fit.tbl <- getFitTbl(subsmpl=FALSE)
    print(round(fit.tbl, digi=1))

    cat("ZLB subsample\n")
    fit.tbl.subsmpl <- getFitTbl(subsmpl=TRUE)
    print(round(fit.tbl.subsmpl, digi=1))

    if (to.latex) {
        require(xtable)
        latex.tbl <- xtable(fit.tbl, digits=1)
        filename.tbl <- ("tables/fit_fullsmpl.tex")
        cat("*** writing table ", filename.tbl, "\n")
        sink(filename.tbl)
        print(latex.tbl, include.rownames=TRUE, include.colnames=FALSE, only.contents=TRUE, hline.after=NULL)
        sink()

        latex.tbl <- xtable(fit.tbl.subsmpl, digits=1)
        filename.tbl <- ("tables/fit_subsmpl.tex")
        cat("*** writing table ", filename.tbl, "\n")
        sink(filename.tbl)
        print(latex.tbl, include.rownames=TRUE, include.colnames=FALSE, only.contents=TRUE, hline.after=NULL)
        sink()
    }
}

printViolations <- function(models, to.latex=FALSE) {
    ## print violations of ZLB
    ## violations of ZLB by expected/forward policy path of the affine model
    ## Table 2
    printStats <- function(stats.viol) {
        tmp <- cbind(dates, stats.viol)
        if (any(stats.viol[,1]>0)) {
            tmp <- tmp[stats.viol[,1]>0,]
            ## display: model, number of violations, first date/last date of violations, average horizon of below zero expectations
            if (is.matrix(tmp)) {
                cat(nrow(tmp), min(tmp[,1]), max(tmp[,1]), round(mean(tmp[,2]),digi=1), "\n")
            } else {
                cat(1, tmp, "\n")
            }
        } else {
            cat("no violations\n")
        }
    }
    T <- length(dates)
    for (m in models) {
        if (m$flag.zlb==T)
            stop("printViolations should be called for AFFINE MODELS")
        stats.forw.viol <- matrix(0, T, 2);  ## number of periods, average horizon
        stats.exp.viol <- matrix(0, T, 2);  ## number of periods, average horizon
        for (t in 1:T) {
            ## forward curve
            path.hat <- getMeanAffine(m$cP[t,], m$muQ, m$PhiQ, m$rho0.cP, m$rho1.cP)
            if (any(path.hat<0))
                stats.forw.viol[t,] <- c(sum(path.hat<0), mean(which(path.hat<0)))
            ## expected short rates
            path.rn <- getMeanAffine(m$cP[t,], m$mu, m$Phi, m$rho0.cP, m$rho1.cP)
            if (any(path.rn<0))
                stats.exp.viol[t,] <- c(sum(path.rn<0), mean(which(path.rn<0)))
        }
        cat(getModelName(m), "\n")
        cat("forward curve violations:\n")
        printStats(stats.forw.viol)
        cat("expectations violations:\n")
        printStats(stats.exp.viol)
    }
    if (to.latex==T)
        stop("TODO: create tex file with Table 2")
}

plotViolations <- function(models, plot.mode=0, export=FALSE) {
    ## plot violations of ZLB
    ## -> probabilities of negative future short rates
    t1 <- which(dates==20000131)
    t2 <- length(dates)
    yrange <- c(0,1)
    ltys <- c(1,2,3);
    lwds <- c(1,1,1);
    colors = c("black", "black", "black")
    if (plot.mode!=1) {
        openPlotDevice(plot.mode, filename = "prob_neg")
        par(mar = c(4,4,2,2)+.1, mfrow = c(2,ceiling(length(models)/2)))
    }
    for (i in seq_along(models)) {
        m <- models[[i]]
        if (plot.mode==1) {
            openPlotDevice(plot.mode, filename=paste("prob_neg", m$name, sep="_"))
            par(mar = c(4,4,1,1)+.1)
        }
        if (m$flag.zlb==T)
            stop("printViolations should be called for AFFINE MODELS")
        w <- ts(matrix(NA, t2-t1+1, 3), start=c(2000,1), frequency=12)  ## for three different horizons
        for (t in t1:t2) {
            mutph <- getMeanAffine(m$cP[t,], m$mu, m$Phi, m$rho0, m$rho1, H=24)
            sigtph <- sqrt(getVarAffine(m$Phi, m$Omega, m$rho1, H=24))
            w[t-t1+1,1] <- pnorm(-mutph[6]/sigtph[6])   # different future horizons
            w[t-t1+1,2] <- pnorm(-mutph[12]/sigtph[12])
            w[t-t1+1,3] <- pnorm(-mutph[24]/sigtph[24])
        }
        plot(w[,1], type="l", ylim=yrange, lty=ltys[1], lwd=lwds[1], col=colors[1], xlab="Years", ylab="Probability", xaxs="i", yaxs="i")
        if (plot.mode!=1)
            title(paste("Model", getModelName(m)))
        plot.recessions(yrange)
        lines(w[,1], lty=ltys[1], lwd=lwds[1], col=colors[1])
        lines(w[,2], lty=ltys[2], lwd=lwds[2], col=colors[2])
        lines(w[,3], lty=ltys[3], lwd=lwds[3], col=colors[3])
        if (i==1 | plot.mode==1)
            legend("topleft", c("six months", "one year", "two years"), lwd=lwds, col=colors, lty=ltys, bg="white")
        if (plot.mode==1)
            dev.off()
        if (export) {
            colnames(w) <- c("6m", "1y", "2y")
            df <- data.frame(dates=dates[t1:t2], prob=w)
            write.csv(df, paste0("export/fig1_prob_af_", names(models)[i], ".csv"))
        }

    }
    if (plot.mode==2)
        dev.off()
}

getYieldForecasts <- function(X, mu, Phi, Sigma, rho0, rho1, muQ, PhiQ, H, mat, rmin, flag.zlb, zlb.method = c("Krippner", "CGF", "MC"), zlb.approx = TRUE, Nsim = 1000, median.forecast=FALSE) {
    ## calculate yield forecasts
    ## Arguments:
    ##   X - time series of risk factors    TxN
    ##   mu, Phi, Sigma, rho0, rho1, muQ, PhiQ -  model parameters
    ##   H - maximum forecast horizon
    ##   mat - maturity of yield to forecast
    ##   rmin - lower bound for yields
    ##   flag.zlb - ZLB model or affine model?
    ##   zlb.method - method to calculate shadow-rate yields
    ##   zlb.approx - should E_t(f(X[t+h]) be approximated by f(E_t(X[t+h])) ?
    ##   Nsim - (used if zlb.approx = FALSE) number of Monte Carlo simulations
    ## Value:
    ##   forecast yields - TxH-matrix -- |E_t y_t+h| t=1:T, h=1:H
    ## Globals:
    ##   mats

    zlb.method <- match.arg(zlb.method)
    N <- length(rho1)
    if (!is.matrix(X) || ncol(X)!=N)
        stop("X does not have right dimension")
    T <- nrow(X)
    Ey <- matrix(NA, T, H)
    Omega <- Sigma %*% t(Sigma)
    if (missing(rmin) || is.null(rmin))
        rmin <- 0
    if (flag.zlb) {
        if ((!zlb.approx) & !(zlb.method=="Krippner")) {
            zlb.method <- "Krippner"
            warning("getYieldForecasts: Monte Carlo forecasts for shadow-rate model -- zlb.method changed to 'Krippner'", call.=FALSE)
        }
        if (zlb.method == "CGF")
            getYfn <- make.getY.CGF(rho0, rho1, muQ, PhiQ, Sigma, rmin=rmin, flag.zlb=TRUE)
        else if (zlb.method == "Krippner")
            getYfn <- make.getY.Krippner(rho0, rho1, muQ, PhiQ, Sigma, rmin=rmin, flag.zlb=TRUE)
        else if (zlb.method == "MC")
            getYfn <- getY.MC(Xt, rho0, rho1, muQ, PhiQ, Sigma, rmin=rmin, flag.zlb=flag.zlb)
        getY <- function(Xt) getYfn(Xt)[mats==mat]
    } else {
        ## affine model
        loads <- gaussian.loadings(mat, muQ, PhiQ-diag(N), Sigma%*%t(Sigma), rho0, rho1)
        ## impose ZLB ad hoc
        getY <- function(Xt) pmax(rmin, as.numeric(loads$A + crossprod(loads$B, Xt)))
    }
    if (!zlb.approx) {
        shocks <- matrix(NA, N, Nsim);
        ind.even <- seq(2,Nsim,2)
        ind.odd <- seq(1,Nsim,2)
    }
    for (t in 1:T) {
        EXtph <- X[t, ]
        VarXtph <- 0
        Phi.h <- diag(N)    # Phi.h contains Phi^(h-1)
        for (h in 1:H) {
            EXtph <- as.numeric(mu + Phi %*% EXtph)
            if (zlb.approx) {
                ## ignore non-linearity in forecast -- as I had done it before
                Ey[t, h] <- getY(EXtph)
            } else {
                ## obtain conditional expectation using Monte Carlo
                ## (1) draw from conditional distribution of risk factors
                ## variance - sum_(i=1)^h Phi_(i-1) * Omega * t(Phi_(i-1))
                VarXtph <- VarXtph + Phi.h %*% Omega %*% t(Phi.h)
                Phi.h <- PhiQ %*% Phi.h    ## Phi^h
                tmp <- matrix(rnorm(Nsim*N/2), N, Nsim/2)
                shocks[, ind.even] <- tmp
                shocks[, ind.odd] <- -tmp   # antithetic sampling
                Xsim <- replicate(Nsim, EXtph) + t(chol(VarXtph)) %*% shocks
                ## (2) calculate yield
                Ysim <- apply(Xsim, 2, getY)
                ## (3) take the meann
                Ey[t, h] <- ifelse(median.forecast, median(Ysim), mean(Ysim))
            }
        }
    }
    Ey
}

printForecastAccuracy <- function(models, to.latex=FALSE, dm.pairs, rmin,
                                  zlb.approx=TRUE, d.loss=1, median.forecast=TRUE, H=24) {
    ## assess forecast accuracy of affine and ZLB models
    ## Globals:
    ##  dates, Y
    if (!(d.loss %in% c(1,2)))
        stop("d.loss must be 1 or 2")
    mat <- 3  ## which interest rate to forecast?
    h.sel <- seq(6,H,6)
    t1 <- which(dates==20081231)
    t2 <- length(dates)-H
    cat("## Forecasting\n")
    cat("Target: yield with maturity", mat, "months\n")
    cat("Horizons: 1 -", H, "months\n")
    cat("First forecast at", dates[t1], "\n")
    cat("Last forecast at", dates[t2], "\n")
    tbl <- matrix(NA, length(models), H)
    rownames(tbl) <- sapply(models, getModelName)
    colnames(tbl) <- 1:H
    FEs <- array(NA, c(length(models), t2-t1+1, H))
    t0 <- t1+20 ## look at these forecasts
    for (i in seq_along(models)){
        m <- models[[i]]
        if (missing(rmin) || is.null(rmin))
            rmin <- m$rmin
        cat("Forecasting using model", m$name, "\n")
        Y.realized <- t(vapply(t1:t2, function(t) Y[(t+1):(t+H), mats==mat], numeric(H)))
        Y.forecast <- getYieldForecasts(m$cP[t1:t2,], m$mu, m$Phi, m$Sigma, m$rho0.cP, m$rho1.cP, m$muQ, m$PhiQ, H=H, mat=mat, rmin=rmin, flag.zlb=m$flag.zlb, zlb.approx=zlb.approx, median.forecast=median.forecast)
        tmp <- rbind(Y[t0+h.sel, mats==mat],
                     Y.forecast[t0-t1+1, h.sel])
        cat("Example forecast:\n")
        rownames(tmp) <- c("Realized yields", "Forecasts")
        colnames(tmp) <- h.sel
        print(round(120000*tmp, digi=0))
        FEs[i,,] <- round(120000*(Y.realized - Y.forecast), digi=2) ## (t2-t1+1) x H  -- in bps for better scaling
        ## debug
        ## rownames(FEs) <- dates[t1:t2]
        ## print(round(120000*FEs[1, h.sel], digi=2))
        ## print(round(120000*sqrt(colMeans(FEs[,h.sel]^2)), digi=2))
        tbl[i, ] <- (colMeans(abs(FEs[i,,])^d.loss))^(1/d.loss)
    }

    cat("#", ifelse(d.loss==1, "Mean absolute error", "Root-mean-squared error"), "in basis points\n")
    print(round(tbl[,h.sel], digi=0))

    if (to.latex) {
        require(xtable)
        filename.tbl <- "tables/forecasts.tex"
        cat("*** writing table ", filename.tbl, "\n")
        sink(filename.tbl)
        latex.table <- xtable(tbl[,h.sel], digits=1)
        print(latex.table, hline.after=NULL, only.contents=T, include.colnames=F)
        sink()
    }

    if (!missing(dm.pairs)) {
        cat("# Diebold-Mariano tests\n")
        require(forecast)
        ##ind <- seq(2, length(models), by=2)
        dm.stats <- matrix(NA, nrow(dm.pairs), H)
        ##rownames(dm.stats) <- sapply(ind, function(i) paste(models[[i-1]]$name, "vs", models[[i]]$name))
        rownames(dm.stats) <- unlist(apply(dm.pairs, 1,
                                           function(p) paste(models[[p[1]]]$name, "vs", models[[p[2]]]$name)))
        colnames(dm.stats) <- 1:H
        dm.pvals <- dm.stats
        for (i in 1:nrow(dm.pairs)) {
            for (h in 1:H) {
                rval <- dm.test2(FEs[dm.pairs[i, 1], , h], FEs[dm.pairs[i, 2], , h], h=h, power=d.loss)
                if (is.na(rval$statistic)) {
                    dm.stats[i, h] <- NA
                    dm.pvals[i, h] <- 0
                } else {
                    dm.stats[i, h] <- rval$statistic
                    dm.pvals[i, h] <- rval$p.value
                }
            }
        }
        print(round(dm.stats[,h.sel], digi=3))
        print(round(dm.pvals[,h.sel], digi=3))

        for (i in 1:nrow(dm.pairs)) {
            m1 <- dm.pairs[i, 1]; m2 <- dm.pairs[i, 2]
            cat(getModelName(models[[m2]]), "/", getModelName(models[[m1]]), ":",
                sprintf("%4.2f", tbl[m2,h.sel]/tbl[m1,h.sel]), "\n")
        }

        if (to.latex) {
            filename.tbl <- "tables/forecasts-dm.tex"
            cat("*** writing table ", filename.tbl, "\n")
            sink(filename.tbl)
            for (i in 1:nrow(dm.pairs)) {
                m1 <- dm.pairs[i, 1]; m2 <- dm.pairs[i, 2]
                cat(getModelName(models[[m2]]), "/", getModelName(models[[m1]]), sep="")
                for (h in h.sel) {
                    cat(sprintf("& %4.2f", tbl[m2,h]/tbl[m1,h]),
                        if (dm.pvals[i, h] < 0.01) {
                            "$^{\\ast\\ast\\ast}$"
                        } else if (dm.pvals[i, h] < 0.05) {
                            "$^{\\ast\\ast}$"
                        } else if (dm.pvals[i, h] < 0.1) {
                            "$^\\ast$"
                        }, sep="")
                }
                cat("\\\\ \n")
            }
            sink()
        }

    }
}

dm.test2 <- function (e1, e2, alternative = c("two.sided", "less", "greater"),
                      h = 1, power = 2)
{
    ## improved version of dm.test from package "forecast"
    alternative <- match.arg(alternative)
    d <- c(abs(e1))^power - c(abs(e2))^power
    d.cov <- acf(d, na.action = na.omit, lag.max = h - 1, type = "covariance",
        plot = FALSE)$acf[, , 1]
    d.var <- sum(c(d.cov[1], 2 * d.cov[-1]))/length(d)
    dv <- d.var
    STATISTIC <- mean(d, na.rm = TRUE)/sqrt(dv)
    names(STATISTIC) <- "DM"
    if (alternative == "two.sided")
        PVAL <- 2 * pnorm(-abs(STATISTIC))
    else if (alternative == "less")
        PVAL <- pnorm(STATISTIC)
    else if (alternative == "greater")
        PVAL <- pnorm(STATISTIC, lower.tail = FALSE)
    PARAMETER <- c(h, power)
    names(PARAMETER) <- c("Forecast horizon", "Loss function power")
    structure(list(statistic = STATISTIC, parameter = PARAMETER,
        alternative = alternative, p.value = PVAL, method = "Diebold-Mariano Test",
        data.name = c(deparse(substitute(e1)), deparse(substitute(e2)))),
        class = "htest")
}

openPlotDevice <- function(plot.mode=0, filename, eps.width=7, eps.height=8.5, pdf.width=7, pdf.height=5) {
    if (plot.mode==0) {
        dev.new()
    } else {
        if (missing(filename))
            stop("filename can't be missing if plotting to file")
        filename <- paste("figures/", filename, sep="")
        if (plot.mode==1) {
            filename <- paste(filename, ".pdf", sep="")
            ## for presentation
            pdf(filename, width=pdf.width, height=pdf.height, pointsize=12)
        } else if (plot.mode==2) {
            filename <- paste(filename, ".eps", sep="")
            ## for paper
            postscript(filename, horizontal=FALSE, onefile=FALSE, paper="special", width=eps.width, height=eps.height)
        }
        cat("saving plot in file:", filename, "\n")
    }
}

plot.recessions <- function(yrange) {
    rec.90.from <- 1990+7/12
      rec.90.to <- 1991+3/12
      rec.01.from <- 2001+3/12
      rec.01.to <- 2001+11/12
      rec.07.from <- 2007+12/12
      rec.07.to <- 2009+6/12
      polygon(x=c( rec.90.from, rec.90.from, rec.90.to, rec.90.to),
                        y=c(yrange, rev(yrange)),
                        density=NA, col="gray", border=NA)
      polygon(x=c( rec.01.from, rec.01.from, rec.01.to, rec.01.to),
                        y=c(yrange, rev(yrange)),
                        density=NA, col="gray", border=NA)
      polygon(x=c( rec.07.from, rec.07.from, rec.07.to, rec.07.to),
                        y=c(yrange, rev(yrange)),
                        density=NA, col="gray", border=NA)
}


plotData <- function(plot.mode) {
    start.date <- c(1985, 1)
    y120 <- ts( Y[,mats==120]*1200, start=start.date, frequency=12)
    y24 <- ts( Y[,mats==24]*1200, start=start.date, frequency=12)
    yshort <- ts( Y[,1]*1200, start=start.date, frequency=12)

    plotYields <- function() {
        yrange <- range(0, y120,y24,yshort)
        lwds <- c(2,2,2)
        ltys <- c(1,2,3)
        colors <- c("black", "green", "blue")
        plot(yshort,  type="l", xlab="Year", ylab="Percent", ylim=yrange, col=colors[1], lwd=lwds[1], lty=ltys[1],
             xaxs="i", yaxs="i")
        plot.recessions(yrange)
        lines(yshort, col=colors[1], lwd=lwds[1], lty=ltys[1])
        lines(y24, col=colors[2], lwd=lwds[2], lty=ltys[2])
        lines(y120, col=colors[3], lwd=lwds[3], lty=ltys[3])
        legend("topright", c("3m T-bill","2y yield", "10y yield"), lwd=lwds, col=colors, lty=ltys, bg="white", cex=.8)
    }

    plotMacro <- function() {
        ## observed macro variables
        inf.obs <-  ts(M.o[,1]*1200, start=start.date,frequency=12)
        ugap.obs <- ts(M.o[,2]*1200, start=start.date,frequency=12)
        yrange <- range(0, inf.obs, ugap.obs)
        lwds <- c(2,2)
        ltys <- c(1,2)
        colors <- c("red", "blue")
        plot(inf.obs, type="l", xlab="Year", ylab="Percent", ylim=yrange, col=colors[1], lwd=lwds[1], lty=ltys[1],
             xaxs="i", yaxs="i")
        plot.recessions(yrange)
        lines(inf.obs, col=colors[1], lwd=lwds[1], lty=ltys[1])
        lines(ugap.obs, col=colors[2], lwd=lwds[2], lty=ltys[2])
        legend("topright", c("inflation", "unemployment gap"), lwd=lwds, col=colors, lty=ltys, bg="white", cex=.8)
    }

    if (plot.mode==1) {
        ## presentation -- separate figures
        openPlotDevice(plot.mode, filename = "data_yields")
        par(mar = c(4,4,1,1)+.1)
        plotYields()
        dev.off()
        openPlotDevice(plot.mode, filename = "data_macro")
        par(mar = c(4,4,1,1)+.1)
        plotMacro()
        dev.off()
    } else {
        ## paper/screen -- yields top panel, macro bottom panel
        openPlotDevice(plot.mode, filename = "data")
        par(mar = c(4,4,2,2)+.1, mfrow = c(2,1))
        plotYields()
        plotMacro()
        if (plot.mode>0)
            dev.off()
    }

}

plotShadowRates <- function(models1, models2, start.date = 20050131,
                            plot.mode=0, flag.taylor=FALSE, filename = "shadow_rates") {
    ## plot time series of shadow rates in one graph
    ## Arguments:
    ##   models - list with model shadow rates
    ##   plot.mode - 0 (default) screen, 1 for paper, 2 for presentation
    ## Globals:
    ##   uses Y, dates
    require(zoo)

    plotFn <- function(models, title.str="") {
        M <- length(models)
        yrange <- range( sapply(models, function(model) model$s[ind]), Y[ind,1])*1200
        ltys <- c(1,1,5,2,4,3,6);
        ##ltys <- c(1,1,1,1,1,1,1,1,1,1)
        lwds <- c(1,2,2,2,2,2,2,2,2,2);
        colors = c("black", "blue", "green", "red", "black", "gray", "brown", "gray", "yellow", "blue")
        tbill <- ts(Y[ind,1]*1200, start=start.time, frequency=12)
        plot(tbill, type="l", ylim=yrange, lty=ltys[1], lwd=lwds[1], col=colors[1], xlab="Years", ylab="Percent", xaxs="i", yaxs="i")
        plot.recessions(yrange)
        title(title.str)
        lines(tbill, lty=ltys[1], lwd=lwds[1], col=colors[1])
        for (m in 1:M) {
            s.plot <- ts(models[[m]]$s[ind]*1200, start=start.time, frequency=12)
            lines(s.plot, lty=ltys[m+1], lwd=lwds[m+1], col=colors[m+1])
        }
        abline(h=0,lty=5)
        if (flag.taylor) {
            T.smpl <- dates<20080000  ## estimation subsample
            lm.taylor <- lm(Y[T.smpl,1]*n.per*100 ~ M.o[T.smpl,])
            r.taylor <- ts(cbind(1, M.o[ind,]) %*% lm.taylor$coef, start=start.time, frequency=12)
            lines(r.taylor, lty=ltys[2+M], lwd=lwds[2+M], col=colors[2+M])
            legend("topright", c("3m T-Bill", sapply(models, getModelName), "Taylor rule"), lwd=lwds[1:(M+2)], col=colors[1:(M+2)], lty=ltys[1:(M+2)], bg="white", cex=0.9)
        } else {
            legend("bottomleft", c("3m T-Bill", sapply(models, getModelName)), lwd=lwds, col=colors, lty=ltys, bg="white", cex=0.8)
        }
    }

    ind <- dates>=start.date
    start.time <- as.numeric(as.yearmon(convertDate(start.date)))
    if (plot.mode==1) {
        ## presentation: plot separate figures
        openPlotDevice(1, "shadow_rates_1")
        par(mar = c(4,4,1,1))
        plotFn(models1)
        dev.off()
        openPlotDevice(1, "shadow_rates_2")
        par(mar = c(4,4,1,1))
        plotFn(models2)
    } else {
        ## paper/screen: plot two panels
        openPlotDevice(plot.mode, filename,
                       eps.height=9)
        par(mar = c(4,4,2,2)+.1, mfrow=c(2,1))
        plotFn(models1, "Model YZ(3)")
        plotFn(models2, "Model MZ(2)")
    }
    if (plot.mode>0)
        dev.off()
}

plotShortRateDist <- function(model, plot.mode=0, date=max(dates), h=4*n.per) {
    ## plot distribution of future short rates / shadow rates
    ## Arguments:
    ##   model - model parameters and risk factors -- only one model here
    ##   date  - sample date of interest
    ##   h     - future horizon

    t0 <- which(dates==date)
    ## conditional moments of future shadow rate
    mutph <- getMeanAffine(model$cP[t0,], model$muQ, model$PhiQ, model$rho0.cP, model$rho1.cP)
    sig2tph <- getVarAffine(model$PhiQ, model$Omega, model$rho1.cP)
    Ertph <- getMeanBlack(mutph=mutph, sig2tph=sig2tph, rmin=model$rmin)

    r.grid <- seq(-4,6,.01)/1200
                                        #r.grid <- seq(floor(mutph[h]-2), ceiling(mutph[h]+2), .01)/1200
    fQs <- 1/sqrt(2*pi*sig2tph[h])*exp(-(r.grid - mutph[h])^2/2/sig2tph[h]) ## future shadow rate
    fQr <- 10000*(r.grid==0) + fQs*(r.grid>0) ## future short rate

    openPlotDevice(plot.mode,
                   filename = paste("densities", model$name, h, date, sep="_"),
                   eps.height=5)
    par(mar = c(4,4,2,2)+.1)
    lwds <- c(1,2)
    ltys <- c(2,1)
    colors <- c("blue", "black")
    yrange <- range(fQs/1200+.01,0)
    plot(r.grid*1200, fQs/1200, type='l', xlab="Percent", ylab="Density", ylim=yrange, col=colors[1], lwd=lwds[1], lty=ltys[1], xaxs="i", yaxs="i")
    lines(r.grid[r.grid>=0]*1200, fQr[r.grid>=0]/1200, col=colors[2], lwd=lwds[2], lty=ltys[2])
    abline(v=mutph[h]*1200, lty=2)
    abline(v=Ertph[h]*1200, lty=3)
    text(mutph[h]*1200-.1,max(yrange)/3, "mode", adj=c(1,0))
    text(Ertph[h]*1200+.1,max(yrange)/3, "mean", adj=c(0,0))
    legend("topleft", c("density shadow rate", "density short rate"), lwd=lwds, col=colors, lty=ltys, bg="white")
    if (plot.mode>0)
        dev.off()
}

printMeans <- function(model) {
    ## print unconditional means of short rate
    N <- length(model$mu)
    cat("Q-mean = ", 100*n.per*(model$rho0.cP + crossprod(model$rho1.cP, solve(diag(N) - model$PhiQ) %*% model$muQ)), "\n")
    cat("P-mean (pop)  = ", 100*n.per*(model$rho0.cP + crossprod(model$rho1.cP, solve(diag(N) - model$Phi) %*% model$mu)), "\n")
    cat("P-mean (smpl) = ", 100*n.per*(model$rho0.cP + crossprod(model$rho1.cP, colMeans(model$cP))), "\n")
}

printPersistence <- function(model, i=1) {
    ## print persistence statistics
    ## Arguments:
    ##  model - list with parameters
    ##  i - which factor for IRF
    cat("eigenvalues under P = ")
    cat(abs(eigen(model$Phi)$values))
    cat("\n")
    cat("eigenvalues under Q = ")
    cat(1+model$lamQ)
    cat("\n")
    cat("IRF(5y) under P = ")
    h.irf <- 5*n.per
    cat(irf.var1(model$Phi, max.lag=h.irf, g=i, h=i)[h.irf])
    cat("\n")
    cat("IRF(5y) under Q = ")
    N <- length(model$mu)
    cat(irf.var1(model$PhiQ, max.lag=h.irf, g=i, h=i)[h.irf])
    cat("\n")
}

getMeanAffine <- function(Xt, mu, Phi, rho0, rho1, H=120, return.current=FALSE) {
    ## Conditional mean of affine short rate - E_t s_t+h
    ## Arguments:
    ##   return.current - choose whether h=1:H (FALSE - default), or h=0:H (TRUE)

    EXtph <- Xt
    EX <- matrix(NA, length(mu), H+1)
    EX[,1] <- Xt
    Er <- numeric(H+1)
    Er[1] <- as.numeric(rho0 + crossprod(rho1, Xt))
    for (h in 1:H) {
        ##    for (h in 1:H) {
        EXtph <- mu + Phi %*% EXtph
        Er[1+h] <- as.numeric(rho0 + crossprod(rho1, EXtph))
        EX[,1+h] <- EXtph
    }
    ##print(head(cbind(t(EX), Er), n=24))
    if (return.current) {
        Er
    } else {
        tail(Er, -1)
    }
}

getVarAffine <- function(Phi, Omega, rho1, H=120) {
    ## Get conditional variance (forecast error variance) of affine short rate
    VartZtph <- 0
    sig2tph <- numeric(H)
    N <- length(rho1)
    Phi.h <- diag(N)  ## for VAR(1): Psi_i = Phi^i
    ## Phi.h contains Phi^(h-1)
    ## covariance matrix of h-step-ahead forecast errors is
    ##   sum_(i=1)^h Psi_(i-1) * Omega * t(Psi_(i-1))
    for (h in 1:H) {
        VartZtph <- VartZtph + Phi.h %*% Omega %*% t(Phi.h)
        sig2tph[h] <- t(rho1) %*% VartZtph %*% rho1
        Phi.h <- Phi %*% Phi.h    ## Phi^h
    }
    sig2tph
}

getMeanBlack <- function(Xt, mu, Phi, Omega, rho0, rho1, mutph, sig2tph, H=120, rmin=0) {
    ## Conditional mean of censored short rate
    ## i.e., expected short rate of Black shadow rate model
    if (missing(mutph))
        mutph <- getMeanAffine(Xt, mu, Phi, rho0, rho1, H)
    if (missing(sig2tph))
        sig2tph <- getVarAffine(Phi, Omega, rho1, H)
    rmin + (mutph-rmin)*pnorm((mutph-rmin)/sqrt(sig2tph)) + sqrt(sig2tph)*dnorm((mutph-rmin)/sqrt(sig2tph))
}

plotPaths <- function(m, plot.mode=0, H=60, plot.dates=max(dates), flag.P=TRUE, mean.only=FALSE, filename, export=FALSE) {
    ## plot forward curve and shadow forward curve
    if (missing(filename))
        filename <- paste("paths", m$name, sep="_")
    openPlotDevice(plot.mode, filename)
    par(mar = c(4,4,2,2)+.1, mfrow = c(length(plot.dates),1))
    lwds <- c(1,2,2,2,2,2)
    ltys <- c(1,2,1,3,4,1)
    colors <- c("black", "black", "black", "blue", "blue", "blue")
    yrange <- range(-2, 3)
    cat("**************************************\n")
    cat("Model", getModelName(m), "\n")
    printMeans(m)
    printPersistence(m, i=ifelse(m$flag.macro, 3, 1))
    for (date in plot.dates) {
        t <- which(dates==date)
        ## Q-mean
        Q.mean <- getMeanBlack(m$cP[t,], m$muQ, m$PhiQ, m$Omega, m$rho0.cP, m$rho1.cP, rmin=m$rmin, H=H)
        ## Q-mode
        Q.mean.shadow <- getMeanAffine(m$cP[t,], m$muQ, m$PhiQ, m$rho0.cP, m$rho1.cP, H=H)
        ## model path
        Q.mode <- pmax(Q.mean.shadow, m$rmin)
        plot(1:H, Q.mean*100*n.per, type='l', , xlab="Horizon", ylab="Percent", ylim=yrange, col=colors[1], lwd=lwds[1], lty=ltys[1], xaxs="i", yaxs="i")
        title(format(convertDate(dates[t]), "%B %Y"))
        abline(h=0, lty=5)
        abline(h=.25, col="black", lwd=1, lty=3)
        legend.str <- "Q-mean short rate"
        ## export matrix
        A <- matrix(Q.mean, 1, H)
        if (!mean.only) {
            lines(1:H, Q.mean.shadow*100*n.per, col=colors[2], lwd=lwds[2], lty=ltys[2])
            lines(1:H, Q.mode*100*n.per, col=colors[3], lwd=lwds[3], lty=ltys[3])
            A <- rbind(A, Q.mean.shadow, Q.mode)
            legend.str <- c(legend.str, "Q-mean shadow rate", "Q-mode short rate")
        }
        if (flag.P) {
            P.mean <- getMeanBlack(m$cP[t,], m$mu, m$Phi, m$Omega, m$rho0.cP, m$rho1.cP, rmin=m$rmin, H=H)
            lines(1:H, P.mean*100*n.per, col=colors[4], lwd=lwds[4], lty=ltys[4])
            P.mean.shadow <- getMeanAffine(m$cP[t,], m$mu, m$Phi, m$rho0.cP, m$rho1.cP, H)
            lines(1:H, P.mean.shadow*100*n.per, col=colors[5], lwd=lwds[5], lty=ltys[5])
            P.mode <- pmax(P.mean.shadow, m$rmin)
            lines(1:H, P.mode*100*n.per, col=colors[6], lwd=lwds[6], lty=ltys[6])
            A <- rbind(A, P.mean, P.mean.shadow, P.mode)
            legend.str <- c(legend.str, "P-mean short rate", "P-mean shadow rate", "P-mode short rate")

            lines(1:H, Q.mode*100*n.per, col=colors[3], lwd=lwds[3], lty=ltys[3])
        }
        if (date==max(plot.dates))
            legend("bottomright", legend.str, lwd=lwds, col=colors, lty=ltys, bg="white", cex=0.8)
        rownames(A) <- legend.str
        colnames(A) <- paste0(1:H, "m")
        A <- A*1200
        if (export)
            write.csv(A, paste0("export/fig3_paths_", date, ".csv"))
    }
    if (plot.mode>0)
        dev.off()
}

plotShadowYields <- function(m, plot.mode=0, plot.dates=max(dates)) {
    ## plot fitted and shadow yield curve
    ## Arguments:
    ##  models - list with fitted and shadow yields
    ##  ...
    ## Globals: uses mats

    cat("## shadow yields\n")

    openPlotDevice(plot.mode,
                   filename=paste("shadow_yields", m$name, sep="_"))

    par(mar = c(4,4,2,2)+.1)
    if (plot.mode==1) {
        par(mfrow = c(1, length(plot.dates)))
    } else {
        par(mfrow = c(length(plot.dates), 1))
    }
    ltys <- c(1,1,2);
    lwds <- c(2,2,2);
    colors = c("black", "black", "red")
    yrange <- range(-2, 4)
    for (date in plot.dates) {
        t <- which(dates==date)
        plot(mats, Y[t,]*100*n.per, type='p', ylim=yrange, xlab='Maturity', ylab='Percent', col=colors[1], lwd=lwds[1], pch=4, xaxs="i", yaxs="i")
        title(format(convertDate(dates[t]), "%B %Y"))
        lines(mats, m$Y.hat[t,]*100*n.per, col=colors[2], lwd=lwds[2], lty=ltys[2])
        lines(mats, m$Y.shadow[t,]*100*n.per, col=colors[3], lwd=lwds[3], lty=ltys[3])
        if (date==max(plot.dates))
            legend("bottomright", c("actual yields", "fitted yield curve", "shadow yield curve"), lwd=lwds, col=colors, pch=c(4,NA,NA), lty=c(NA,ltys[2:3]), bg="white")
        abline(h=0, lty=5)
        ## print difference in yields
        cat("model", m$name, "date", dates[t], ", difference in bps:\n")
        cat(round((m$Y.hat[t,]-m$Y.shadow[t,])*120000, digi=2), "\n")
    }
    if (plot.mode>0)
        dev.off()
}

plotShadowWedgeBAK <- function(m, start.date=20060131, flag.factors=FALSE, mat=120, flag.level=1, plot.mode=0) {
    ## plot time series of shadow yields/level/slope and the difference with observed (the wedge)
    ## Arguments:
    ##  models - list with fitted and shadow yields
    ##  start.date -
    ## Globals: uses mats
    require(zoo)
    sel <- dates>=start.date
    start.time <- as.numeric(as.yearmon(convertDate(start.date)))
    j <- which(mats==mat)

    if (flag.factors) {
        ## normalize PCs so that sum of loadings is one
        i <- ifelse(flag.level, 1, 2)
        w <- m$W[i,]
        wnorm <- w/sum(w)
        print(wnorm)
        y <- ts(Y[sel,]%*%wnorm*1200 , start=start.time, frequency=12)
        ind <- i + ifelse(m$flag.macro, 2, 0)
        yshadow <- ts(m$cP[sel,ind]/sum(w)*1200, start=start.time, freq=12)
    } else {
        y <- ts(Y[sel,j]*100*n.per, start=start.time, freq=12)
        yshadow <- ts(m$Y.shadow[sel,j]*100*n.per, start=start.time, freq=12)
    }
    wedge <- y - yshadow

    if (plot.mode==1) {
        openPlotDevice(plot.mode, filename="shadow_wedge_1")
        par(mar = c(4,4,1,1)+.1)
    } else {
        openPlotDevice(plot.mode, filename="shadow_wedge")
        par(mfrow = c(2,1))
        par(mar = c(4,4,2,2)+.1)
    }

    ltys <- c(1,1,1,1,1,1,1,1,1);
    lwds <- c(2,2,2,2,2,2,2,2,2);
    colors = c("blue", "black", "red", "brown", "gray", "yellow", "red", "green", "blue")

    ## top panel: plot actual and shadow yields/factors
    yrange <- range(0, y, yshadow)
    plot(y, ylim=yrange, xlab='Year', ylab='Percent', col='black', lwd=2, lty=2, xaxs="i", yaxs="i")
    legend.str <- 'Observed'
    if (plot.mode!=1)
        title(ifelse(flag.factors, switch(i, "Shadow level", "Shadow slope"), "Shadow yield vs. observed yield"))
    lines(yshadow, col=colors[1], lwd=lwds[1], lty=ltys[1])
    legend("bottomleft", c("Observed yield", "Shadow yield"), lwd=c(2, lwds[1]),
           col=c('black', colors[1]),
           lty=c(2, ltys[1]), bg="white", cex=.9)

    ## bottom panel: plot wedge
    if (plot.mode==1) {
        dev.off()
        openPlotDevice(plot.mode, filename="shadow_wedge_2")
        par(mar = c(4,4,1,1)+.1)
    }
    yrange <- range(0, wedge)
    plot(wedge, ylim=yrange, xlab='Year', ylab='Percent', col=colors[1], lwd=lwds[1], xaxs="i", yaxs="i")
    if (plot.mode!=1)
        title("ZLB wedge")
    abline(h=0, lty=2)

    if (plot.mode>0)
        dev.off()
}

plotShadowWedge <- function(m, plot.mode=0, start.date=20060131, mat=120) {
    ## plot time series of shadow yields and the difference with fitted (the wedge)
    ## Arguments:
    ##  models - list with fitted and shadow yields
    ##  start.date -
    ## Globals: uses mats
    require(zoo)
    sel <- dates>=start.date
    start.time <- as.numeric(as.yearmon(convertDate(start.date)))
    j <- which(mats==mat)
    y <- ts(m$Y.hat[sel,j]*100*n.per, start=start.time, freq=12)
    yshadow <- ts(m$Y.shadow[sel,j]*100*n.per, start=start.time, freq=12)
    wedge <- y - yshadow
    if (plot.mode==1) {
        openPlotDevice(plot.mode, filename="shadow_wedge_1")
        par(mar = c(4,4,1,1)+.1)
    } else {
        openPlotDevice(plot.mode, filename="shadow_wedge")
        par(mfrow = c(2,1))
        par(mar = c(4,4,2,2)+.1)
    }
    ltys <- c(1,1,1,1,1,1,1,1,1);
    lwds <- c(2,2,2,2,2,2,2,2,2);
    colors = c("blue", "black", "red", "brown", "gray", "yellow", "red", "green", "blue")

    ## top panel: plot actual and shadow yields/factors
    yrange <- range(0, y, yshadow)
    plot(y, ylim=yrange, xlab='Year', ylab='Percent', col='black', lwd=2, lty=2, xaxs="i", yaxs="i")
    if (plot.mode!=1)
        title("Shadow yield vs. fitted yield")
    lines(yshadow, col=colors[1], lwd=lwds[1], lty=ltys[1])
    legend("bottomleft", c("Fitted yield", "Shadow yield"), lwd=c(2, lwds[1]),
           col=c('black', colors[1]),
           lty=c(2, ltys[1]), bg="white")

    ## bottom panel: plot wedge
    if (plot.mode==1) {
        dev.off()
        openPlotDevice(plot.mode, filename="shadow_wedge_2")
        par(mar = c(4,4,1,1)+.1)
    }
    yrange <- range(0, wedge)
    plot(wedge, ylim=yrange, xlab='Year', ylab='Percent', col=colors[1], lwd=lwds[1], xaxs="i", yaxs="i")
    if (plot.mode!=1)
        title("ZLB wedge")
    abline(h=0, lty=2)

    if (plot.mode>0)
        dev.off()
}


getPathLiftoff <- function(path, r.lb=0.25) {
    ## return duratin of ZLB period and lift-off date
    ## Arguments:
    ##  path -- expected/forward short/shadow rate path
    ##  forecast.date -- date from which expectations are calculated, "today"
    ## Value: list
    ##  duration -- expected duration in months
    ##  forecast.month -- month of forecast date - YYYYMM
    ##  month -- month of lift-off -- YYYYMM

    ind.above <- path*1200>r.lb
    if (any(!ind.above) && any(ind.above)) {
        ## some above, some below
        max(which(!ind.above))+1
    } else if (all(ind.above)) {
        ## has already lifted off
        0
    } else {
        ## never lifts off (all below)
        NA
    }
}

getLiftoff <- function(model, liftoff.dates, flag.modal = TRUE, flag.P = FALSE) {
    ## calculate estimated liftoff
    ## Globals: dates
    if (flag.P) {
        mu <- model$mu; Phi <- model$Phi
    } else {
        mu <- model$muQ; Phi <- model$PhiQ
    }
    vapply(match(liftoff.dates, dates), function(t)
           if (flag.modal) {
               getPathLiftoff(getMeanAffine(model$cP[t,], mu, Phi, model$rho0, model$rho1))
           } else {
               getPathLiftoff(getMeanBlack(model$cP[t,], mu, Phi, model$Omega, model$rho0, model$rho1))
           },
           numeric(1))
}

analyzeLiftoff <- function(models) {
    ## estimate liftoff for all models, compare, show correlation
    cat("## Liftoff across models -- modal path\n")
    for (flag.P in c(FALSE, TRUE)) {
        cat(ifelse(flag.P, "P", "Q"), "measure\n")
        liftoff <- sapply(models, function(m) getLiftoff(m, dates[zlb.ind], flag.modal=TRUE, flag.P=flag.P))
        colnames(liftoff) <- names(models)
        rownames(liftoff) <- dates[zlb.ind]
        ## cat("Liftoff:\n")
        ## print(tail(liftoff, n=36))
        cat("Correlations of liftoff across models --", min(dates[zlb.ind]), "to", max(dates[zlb.ind]), "\n")
        print(round(cor(liftoff), digi=3))
    }
}

scatterLiftoff <- function(m1, m2, plot.mode=0, filename="scatterLiftoff") {
    openPlotDevice(plot.mode, filename, eps.width=7, eps.height=4)
    par(mar=c(4,4,2,1), mfrow=c(1,2))
    for (flag.P in c(FALSE, TRUE)) {
        l1 <- getLiftoff(m1, dates[zlb.ind], flag.modal=TRUE, flag.P=flag.P)
        l2 <- getLiftoff(m2, dates[zlb.ind], flag.modal=TRUE, flag.P=flag.P)
        cat("Compare liftoff for models", m1$name, "and", m2$name, "\n")
        cat(ifelse(flag.P, "P", "Q"), "measure\n")
        cat("Correlation:", cor(l1, l2), "\n")
        plot(l1, l2, xlim=c(0,50), ylim=c(0,50),
             xlab=paste("Liftoff", m1$name),
             ylab=paste("Liftoff", m2$name))
        abline(a=0, b=1)
        title(paste(ifelse(flag.P, "P", "Q"), "measure"))
    }
    if (plot.mode>0)
        dev.off()
}

simLiftoffDist <- function(Xt, mu, Phi, Omega, rho0, rho1, M=10000, r.lb=0.25, H=600) {
    ## distribution of possible liftoff horizons
    ##  -> alternative definition of liftoff: stay above r.lb for one year
    ## Arguments:
    ##  mu, Phi, Omega -- VAR parameters for factor dynamics
    ##  Xt           -- risk factors at t
    ##  rho0, rho1     -- loadings of short rate on risk factors
    ## Value:
    ##  distribution across horizons
    ## Globals:
    ##  uses n.per
    N <- length(mu)
    rho1 <- as.numeric(rho1)
    mu <- as.numeric(mu)
    Sigma <- t(chol(Omega))
    durations <- numeric(M)
    req.periods <- 12
    s.0 <- 100*n.per*(rho0 + crossprod(rho1, Xt))
    for (i in 1:M) {
        h <- 0
        count.above <- as.numeric(s.0 > r.lb)
        cP <- Xt
        while (count.above < req.periods) {
            h <- h+1
            if (h>H)
                break
            cP <-  mu + Phi %*% cP + Sigma %*% rnorm(N)
            s.t <- 100*n.per*(rho0 + crossprod(rho1, cP))
            count.above <- count.above + as.numeric(s.t > r.lb)
        }
        if (h>H) {
            durations[i] <- NA
        } else {
            durations[i] <- h-req.periods+1
        }
    }

    ## get distribution
    return(list(durations = durations,
                mode = Mode(durations),
                median = median(durations, na.rm=TRUE),
                lq = quantile(durations, probs=0.25, na.rm=TRUE),
                uq = quantile(durations, probs=0.75, na.rm=TRUE),
                mean = mean(durations, na.rm=TRUE)))
}

simPaceDist <- function(Xt, mu, Phi, Omega, rho0, rho1, M=10000, r.lb=0.25, H=1000) {
    ## distribution of pace of tightening (increase over two years after liftoff)
    ## Arguments:
    ##  mu, Phi, Omega -- VAR parameters for factor dynamics
    ##  Xt           -- risk factors at t
    ##  rho0, rho1     -- loadings of short rate on risk factors
    ## Value:
    ##  distribution
    N <- length(mu)
    rho1 <- as.numeric(rho1)
    mu <- as.numeric(mu)
    Sigma <- t(chol(Omega))
    pace <- numeric(M)
    s.0 <- 100*n.per*(rho0 + crossprod(rho1, Xt))
    for (i in 1:M) {
        h <- 0
        cP <- Xt
        h.lift <- NULL
        while (is.null(h.lift) || h < h.lift + 24) {
            h <- h+1
            if (h>H)
                break
            cP <-  mu + Phi %*% cP + Sigma %*% rnorm(N)
            s.t <- 100*n.per*(rho0 + crossprod(rho1, cP))
            if (is.null(h.lift) && (s.t > r.lb))
                h.lift <- h
        }
        if (h>H) {
            pace[i] <- NA
        } else {
            pace[i] <- s.t - 0.25
        }
    }

    ## get distribution
    return(list(pace = pace,
                mode = Mode(pace),
                median = median(pace, na.rm=TRUE),
                lq = quantile(pace, probs=0.25, na.rm=TRUE),
                uq = quantile(pace, probs=0.75, na.rm=TRUE),
                mean = mean(pace, na.rm=TRUE)))
}

## simLiftoffDist.new <- function(Xt, mu, Phi, Omega, rho0, rho1, M=10000, r.liftoff=0.25, H=1200) {
##     ## distribution of liftoff horizons
##     ## Arguments:
##     ##  mu, Phi, Omega -- VAR parameters for factor dynamics
##     ##  cP             -- risk factors at t
##     ##  rho0, rho1     -- loadings of short rate on risk factors
##     ## Value:
##     ##  distribution across horizons
##     ## Globals:
##     ##  uses n.per
##     N <- length(mu)
##     rho1 <- as.numeric(rho1)
##     mu <- as.numeric(mu)
##     Sigma <- t(chol(Omega))
##     durations <- numeric(M)

##     s.t <- 100*n.per*(rho0 + crossprod(rho1, Xt))

##     if (s.t > r.liftoff) {
##         ## liftoff has already occured -- no simulation
##         durations[1:M] <- 0
##     } else {
##         for (i in 1:M) {
##             h <- 0
##             cP <- Xt
##             s <- s.t
##             while (s < r.liftoff) {
##                 h <- h+1
##                 if (h>H)
##                     break
##                 cP <-  mu + Phi %*% cP + Sigma %*% rnorm(N)
##                 s <- 100*n.per*(rho0 + crossprod(rho1, cP))
##             }
##             if (h>H) {
##                 durations[i] <- H+1
##             } else {
##                 durations[i] <- h
##             }
##         }
##     }

##     ## get distribution
##     return(list(durations = durations,
##                 mode = Mode(durations),
##                 median = median(durations),
##                 lq = quantile(durations, probs=0.25),
##                 uq = quantile(durations, probs=0.75),
##                 mean = mean(durations)))
## }

plotLiftoffDist <- function(m, plot.mode=0, date=max(dates), H=120) {
    ## obtain liftoff distribution for a given sample date using Monte Carlo simulation and plot it
    t <- which(dates==date)
    if ('liftoff.distQ' %in% names(m)) {
        t0 <- min(which(post.2007)) ## first obs with liftoff.dist
        ind <- t-t0+1
        dist <- m$liftoff.distQ[[ind]]
    } else {
        set.seed(616)
        dist <- simLiftoffDist(m$cP[t,], m$muQ, m$PhiQ, m$Omega, m$rho0.cP, m$rho1.cP)
    }

    openPlotDevice(plot.mode, filename = paste("liftoffDist", date, m$name, sep="_"),
                   eps.width=7, eps.height=4)
    par(mar = c(4,4,2,2)+.1)

    kdens <- density(dist$durations, n=H+1, from=0, to=H, na.rm=TRUE)
    yrange=range(1.04*kdens$y)
    plot(0:H, kdens$y, type="l", lwd=2, ylim=yrange, xlab="Horizon", ylab="Density", xaxs="i", yaxs="i")
    text(H/2, max(yrange)*.96, paste(
        "mean   = ", round(dist$mean,digi=1), "\n",
        "median = ", dist$median, "\n",
        "mode   = ", dist$mode, "\n",
        "[25%, 75%] = [", dist$lq, ",", dist$uq, "]\n",
        "based on mean path: ", getPathLiftoff(getMeanBlack(m$cP[t,], m$muQ, m$PhiQ, m$Omega, m$rho0.cP, m$rho1.cP, rmin=m$rmin, H=H)), "\n",
        "based on modal path: ", getPathLiftoff(getMeanAffine(m$cP[t,], m$muQ, m$PhiQ, m$rho0.cP, m$rho1.cP, H)), "\n",
        sep=""), adj=c(0,1))
    cat("liftoff distribution,", m$name, "\n")
    print(head(table(dist$durations),5))
    print(tail(table(dist$durations),5))
    if (plot.mode>0)
        dev.off()
}

plotPaceDist <- function(m, plot.mode=0, date=max(dates), H=120) {
    ## obtain liftoff distribution for a given sample date using Monte Carlo simulation and plot it
    t <- which(dates==date)
    simPaceDist_c <- compiler::cmpfun(simPaceDist)
    set.seed(616)
    dist <- simPaceDist_c(m$cP[t,], m$muQ, m$PhiQ, m$Omega, m$rho0.cP, m$rho1.cP, M=50000)

    openPlotDevice(plot.mode, filename = paste("paceDist", date, m$name, sep="_"),
                   eps.width=7, eps.height=4)
    par(mar = c(4,4,2,2)+.1)
    kdens <- density(dist$pace, n=100, from = -2, to = 5, na.rm=TRUE)
    yrange=range(1.04*kdens$y)
    plot(kdens$x, kdens$y, type="l", lwd=2, ylim=yrange, xlab="Increase", ylab="Density", xaxs="i", yaxs="i")
    text(2.6, max(yrange)*.96, paste(
        "mean   = ", round(dist$mean, 2), "\n",
        "median = ", round(dist$median, 2), "\n",
        "mode   = ", round(dist$mode, 2), "\n",
        "[25%, 75%] = [", round(dist$lq, 2), ",", round(dist$uq, 2), "]\n",
        "based on mean path: ", round(getPace(m, date, FALSE, TRUE), 2), "\n",
        "based on modal path: ", round(getPace(m, date, TRUE, TRUE), 2), "\n",
        sep=""), adj=c(0,1))
    cat("pace distribution,", m$name, "\n")
    print(head(table(dist$pace),5))
    print(tail(table(dist$pace),5))
    if (plot.mode>0)
        dev.off()
}

## showSamplePaths <- function(Xt, mu, Phi, Omega, rho0, rho1, M=20, H=120) {
##     ## construct M sample paths of length H
##     ## calculate liftoff for each path
##     ##  - note: overly simple liftoff rule!
##     ##    -> better: wait until 12 months above threshold before calling it
##     ##               (as in simLiftoffDist)

##     N <- length(mu)
##     rho1 <- as.numeric(rho1)
##     mu <- as.numeric(mu)
##     Sigma <- t(chol(Omega))
##     s.t <- 100*n.per*(rho0 + crossprod(rho1, Xt))
##     s <- matrix(NA, M, H)
##     liftoff <- numeric(M)
##     for (i in 1:M) {
##         cP <- Xt
##         for (h in 1:H) {
##             cP <-  mu + Phi %*% cP + Sigma %*% rnorm(N)
##             s[i, h] <- 100*n.per*(rho0 + crossprod(rho1, cP))
##         }
##         liftoff[i] <- min(which(s[i,]>0.25))
##     }
##     out.matrix <- cbind(liftoff, rep(s.t,M), s)
##     colnames(out.matrix) <- c("liftoff", as.character(0:H))
##     print(round(out.matrix[,1:20], digi=2))
## }

getLiftoffDist <- function(m, incl.P = FALSE) {
    simLiftoffDist_c <- compiler::cmpfun(simLiftoffDist)
    set.seed(616)
    cat("Calculating liftoff distribution for each date -- model ", m$name, "\n")
    cat("Q-measure...\n")
    m$liftoff.distQ <- lapply(which(post.2007), function(t)
                             simLiftoffDist_c(m$cP[t,], m$muQ, m$PhiQ, m$Omega, m$rho0.cP, m$rho1.cP))
    for (stat in c("median", "mode", "lq", "uq"))
        m[[paste0("liftoff.distQ.", stat)]] <- ts(vapply(m$liftoff.distQ, function(dist) dist[[stat]], numeric(1)),
                                                  start=c(2008, 1), frequency=12)
    if (incl.P) {
        cat("P-measure...\n")
        m$liftoff.distP <- lapply(which(post.2007), function(t)
            simLiftoffDist_c(m$cP[t,], m$mu, m$Phi, m$Omega, m$rho0.cP, m$rho1.cP))
        for (stat in c("median", "mode", "lq", "uq"))
            m[[paste0("liftoff.distP.", stat)]] <- ts(vapply(m$liftoff.distP, function(dist) dist[[stat]], numeric(1)),
                                                      start=c(2008, 1), frequency=12)
    }
    m
}

## policy events for figures
fg.events <- c(2008+12/12,
               2009+3/12,
               2009+11/12,
               2011+8/12, ## date-based      mid-2013  -- Jul-2013 -- 23 months
               2012+1/12, ## new date-based  late-2014 -- Sep-2014 -- 32 months
               2012+9/12, ## new date-based  mid-2015  -- Jul-2015 -- 34 months
               2012+12/12) ## outcome-based
fg.horizons <- c(23, 32, 34)
fg.events <- fg.events-1/12
## FG1: August 2011 to December 2011: 23..19
fg1 <- ts(seq(23, 19, -1), start=c(2011, 8), freq=12)
## FG2: January 2012 to August 2012: 32..25
fg2 <- ts(seq(32, 25, -1), start=c(2012, 1), freq=12)
## FG3: September 2012 to November 2012: 34..31
fg3 <- ts(seq(34, 31, -1), start=c(2012, 9), freq=12)
lsap.events <- c(2008+12/12, 2009+3/12, ## QE1
                 2010+8/12, ## 2010+9/12, ## QE2
                 2012+8/12, ## , 2012+9/12  ## QE3
                 2012+12/12 ##  ## QE3
                 )
lsap.events <- lsap.events-1/12
cLSAPLWD <- 2; cLSAPLTY <- 1; cLSAPCOL <- 'green'
cFGLWD <- 1; cFGLTY <- 2; cFGCOL <- 'blue'
plotPolicyEvents <- function() {
    ## LSAP guidance events
       for (i in 1:length(lsap.events))
           abline(v=lsap.events[i], lwd=cLSAPLWD, lty=cLSAPLTY, col=cLSAPCOL)
    ## forward guidance events
    for (i in 1:length(fg.events))
        abline(v=fg.events[i], lwd=cFGLWD, lty=cFGLTY, col=cFGCOL)
}

getSurveysPace <- function(start=20080101, end=20141231) {
    ## get survey data
    require(zoo)  # for na.approx
    surveys <- read.csv(getDataPath("surveys_pace.csv"), na.strings="#N/A")
    surveys$date <- as.Date(surveys$Vintage, format="%m/%d/%y")
    ## surveys <- surveys[surveys$date >= convertDate(start), ]
    ## if (!missing(end) && !is.null(end))
    ##     surveys <- surveys[surveys$date>=convertDate(end), ]
    start.ts <- as.numeric(as.yearmon(min(surveys$date)))
    surveys$Pace <- ts(surveys$Pace, start=start.ts, frequency=12)
    surveys$Pace.lin <- na.approx(surveys$Pace, na.rm=FALSE)
    surveys
}

getSurveysLiftoff <- function(start=20080101, end=20141231) {
    ## get survey data
    require(zoo)  # for na.approx
    surveys <- read.csv(getDataPath("surveys_liftoff.csv"), na.strings="#N/A")
    surveys$date <- as.Date(surveys$Vintage, format="%m/%d/%Y")
    surveys <- surveys[surveys$date>=convertDate(start), ]
    if (!missing(end) && !is.null(end))
        surveys <- surveys[surveys$date>=convertDate(end), ]
    start.ts <- as.numeric(as.yearmon(min(surveys$date)))
    surveys$MA <- ts(surveys$MA, start=start.ts, frequency=12)
    surveys$MA.lin <- na.approx(surveys$MA, na.rm=FALSE)
    surveys$PD <- ts(surveys$PD, start=start.ts, frequency=12)
    surveys$PD.lin <- na.approx(surveys$PD, na.rm=FALSE)
    surveys
}

plotLiftoff <- function(m, plot.mode = 0, export=FALSE) {
    ## plot time series of alternative liftoff estimates

    ## prepare plotting
    if (plot.mode==1) {
        openPlotDevice(plot.mode,
                       filename = paste("liftoff", m$name, "1", sep="_"))
    } else {
        openPlotDevice(plot.mode,
                       filename = paste("liftoff", m$name, sep="_"),
                       eps.width=7, eps.height=9)
        par(mfrow=c(2,1))
    }

    liftoff.modal.Q <- ts(getLiftoff(model, dates[post.2007], flag.P = FALSE), start=c(2008, 1), freq=12)
    liftoff.modal.P <- ts(getLiftoff(model, dates[post.2007], flag.P = TRUE), start=c(2008, 1), freq=12)

    ## top panel: model-based estimates
    par(mar = c(4,4,1,1)+.1)
    lwds <- c(2,2,2,2,1,1)
    ltys <- c(1,2,1,2,1,1)
    colors <- c("black", "black", "cyan")
    yrange <- c(0, 60)
    liftoff.mean.Q <- ts(getLiftoff(m, dates[post.2007], flag.modal=FALSE, flag.P=FALSE), start=c(2008, 1), freq=12)
    plot(liftoff.mean.Q, type='l', xlab="", ylab="Months", ylim=yrange, col=colors[1], lwd=lwds[1], lty=ltys[1], xaxs="i", yaxs="i")
    lines(liftoff.modal.Q, col=colors[2], lwd=lwds[2], lty=ltys[2])
    lines(m$liftoff.distQ.median, col=colors[3], lwd=lwds[3], lty=ltys[3])
    if ("liftoff.distQ" %in% names(m)) {
        xx <- seq(from = 2008, len=length(m$liftoff.distQ.lq), by=1/12)
        xx <- c(xx, rev(xx))
        yy <- c(m$liftoff.distQ.lq, rev(m$liftoff.distQ.uq))
        polygon(xx, yy, col='grey', border=NA)
        lines(liftoff.mean.Q, col=colors[1], lwd=lwds[1], lty=ltys[1])
        lines(m$liftoff.distQ.median, col=colors[3], lwd=lwds[3], lty=ltys[3])
        lines(liftoff.modal.Q, col=colors[2], lwd=lwds[2], lty=ltys[2])
    } else {
        warning("plotLiftoff cannot plot liftoff distribution")
    }
    legend("topleft", c("based on mean path under Q ", "based on modal path under Q", "median of liftoff distribution"), lwd=lwds, col=colors, lty=ltys, bg="white", cex=.8)
    abline(h=0, lty=5)

    if (plot.mode==1) {
        dev.off()
        openPlotDevice(plot.mode,
                       filename = paste("liftoff", m$name, "2", sep="_"))
    }

    ## bottom panel
    par(mar = c(4,4,1,1)+.1)
    colors <- c("blue", "black", "darkolivegreen4", "dodgerblue4", "red")
    lwds <- c(2,2,1,1,3)
    ltys <- c(1,2,1,1,1)
    surveys <- getSurveysLiftoff()
    plot(liftoff.modal.P, type='l', , xlab="Year", ylab="Months", ylim=yrange, col=colors[1], lwd=lwds[1], lty=ltys[1], xaxs="i", yaxs="i")
    if ("liftoff.distP" %in% names(m)) {
        xx <- seq(from = 2008, len=length(m$liftoff.distP.lq), by=1/12)
        xx <- c(xx, rev(xx))
        yy <- c(m$liftoff.distP.lq, rev(m$liftoff.distP.uq))
        polygon(xx, yy, col='grey', border=NA)
        lines(liftoff.modal.P, col=colors[1], lwd=lwds[1], lty=ltys[1])
    } else {
        warning("plotLiftoff cannot plot liftoff P-distribution")
    }
    lines(liftoff.modal.Q, col=colors[2], lwd=lwds[2], lty=ltys[2])
    lines(surveys$PD.lin, col=colors[3], lwd=lwds[3], lty=ltys[3])
    lines(surveys$MA.lin, col=colors[4], lwd=lwds[4], lty=ltys[4])
    actual.liftoff <- ts(seq(60, 0, -1), start=c(2010, 12), freq=12)
    lines(actual.liftoff, col="black", lwd=2, lty=1)
    points(surveys$PD, col=colors[3], pch=19)
    points(surveys$MA, col=colors[4], pch=15)
    points(fg.events[4:6], fg.horizons, lwd=lwds[5], lty=ltys[5], col=colors[5], pch=4)
    legend("topleft", c("based on modal path under P", "based on modal path under Q", "Survey of Primary Dealers", "Macroeconomic Advisers", "FOMC forward guidance", "actual liftoff"), lwd=c(lwds, 2), col=c(colors, "black"), lty=c(ltys[1:4], NA, 1), pch=c(NA, NA, 19, 15, 4, NA), bg="white", cex=.8)
    abline(h=0, lty=5)

    if (export) {
        surveys <- surveys[-nrow(surveys),]
        A <- cbind(liftoff.mean.Q, liftoff.modal.Q,
                   m$liftoff.distQ.median, m$liftoff.distQ.lq, m$liftoff.distQ.uq,
                   liftoff.modal.P, m$liftoff.distP.lq, m$liftoff.distP.uq,
                   rbind(matrix(NA, 8, 4), as.matrix(surveys[,c("PD.lin", "PD", "MA.lin", "MA")])))
        rownames(A) <- dates[post.2007]
        colnames(A) <- c("based on Q-mean", "based on Q-mode",
                         "median of liftoff distr. under Q", "LQ of liftoff distr. under Q", "UQ of liftoff distr. under Q",
                         "based on P-mode", "LQ of liftoff distr. under P", "UQ of liftoff distr. under P",
                         "Primary Dealers (lin. interp.)", "Primary Dealers",
                         "Macro Advisers (lin. interp.)", "Macro Advisers")
        write.csv(A, "export/fig6_liftoff.csv")
    }
    if (plot.mode>0)
        dev.off()
}

getPaceMetric <- function(path) {
    liftoff <- getPathLiftoff(path)
    ## liftoff2 <- getPathLiftoff(path, r.liftoff=2)
    ## cat('months until reached two percent:', liftoff2-liftoff, '\n')
    ## increase over two years
    if (is.na(liftoff) || liftoff==0) {
        return(NA)
    } else {
        return(1200*(path[liftoff+24]-path[liftoff]))
    }
}

getPace <- function(model, pace.dates, flag.modal=TRUE, flag.Qmeas=TRUE) {
    ## estimate pace of tightening, based on shadow-rate model
    ## Arguments:
    ##  model - list with model parameters
    ##  pace.dates - vector with dates
    ##  flag.modal - T: use modal path, F: use forward curve
    ##  flag.Qmea  - T: use Q-measure, F: use P-measure
    ## Value:
    ##  vector with pace for each date
    if (flag.Qmeas) {
        mu <- model$muQ; Phi <- model$PhiQ
    } else {
        mu <- model$mu; Phi <- model$Phi
    }
    vapply(pace.dates, FUN.VALUE=numeric(1), FUN=function(date) {
        if (flag.modal) {
            ## expected shadow rates / shadow forward rates
            path <- getMeanAffine(model$cP[dates==date,], mu, Phi, model$rho0.cP, model$rho1.cP)
        } else {
            ## expected short rates / forward rates
            path <- getMeanBlack(model$cP[dates==date,], mu, Phi, model$Omega, model$rho0.cP, model$rho1.cP, rmin=model$rmin)
        }
        getPaceMetric(path)
    })
}

plotPace <- function(m, plot.mode=0, export=FALSE) {
    ## plot expected pace of tightening after liftoff
    cat("Pace of tightening -- increase over two years after liftoff\n")
    pace.Q <- ts(getPace(m, dates[zlb.ind], flag.modal=TRUE, flag.Q=TRUE),
                 start=zlb.start, freq=12)
    pace.P <- ts(getPace(m, dates[zlb.ind], flag.modal=TRUE, flag.Q=FALSE),
                 start=zlb.start, freq=12)
    surveys <- getSurveysPace()

    require(zoo) # for as.yearmon and for linear interpolation
    openPlotDevice(plot.mode,
                   filename = paste("pace", m$name, sep="_"), eps.width=7, eps.height=4)
    par(mar = c(4,4,2,2)+.1)
    lwds <- c(2,2,1)
    ltys <- c(2,1,1)
    colors <- c("blue", "black", "darkolivegreen4")
    yrange <- c(0,3.5)
    plot(pace.P, ylim=yrange, type="l", col=colors[1], lty=ltys[1], lwd=lwds[1], xaxs="i", yaxs="i", xlab="Year", ylab="Increase over two years (pps)")
    lines(pace.Q, col=colors[2], lty=ltys[2], lwd=lwds[2])
    lines(surveys$Pace.lin, col=colors[3], lty=ltys[3], lwd=lwds[3])
    points(surveys$Pace, col=colors[3], pch=19)
    legend("bottomleft", c("modal path under P", "modal path under Q", "Survey of Primary Dealers"), col=colors, lty=ltys, lwd=lwds, pch=c(NA, NA, 19))
    if (plot.mode>0)
        dev.off()
    if (export) {
        A <- cbind(pace.P, pace.Q, surveys$Pace.lin, surveys$Pace)
        rownames(A) <- dates[zlb.ind]
        colnames(A) <- c("under P", "under Q", "Primary Dealers (lin.interp.)", "Primary Dealers")
        write.csv(A, "export/fig7_pace.csv")
    }
}

