##########################################################
## JLS functions -- based on Joslin, Le, Singleton

jls.llk <- function (yields.o, M.o, W, cL, rinfQ, lamQ, gam0, gam1, K0P=NA, K1P=NA, Sigma, mats, dt, sigma.e=NA) {
  ## Compute the likelihood for MTSM
  ## INPUTS:
  ## yields.o   : (T+1)*J,  matrix of observed yields (first row are t=0 observations, which likelihood conditions on)
  ## M.o        : (T+1)*cM  macro factors
  ## cL         : scalar    number of yield factors
  ## W          : cN*J,     vector of portfolio weights to fit without error.
  ## rinfQ      : scalar,   E^Q(r)
  ## lamQ       : cN*1      Q-eigenvalues
  ## gam0       : cM*1      spanning parameters -- intercept
  ## gam1       : cM*cN     spanning parameters -- coefficients
  ## Sigma      : cN*cN,    lower-triangular matrix, Chol decomp of Omega (cov of innovations to Z)
  ## mats       : 1*J,      maturities in years
  ## dt         : scalar,   length of period in years
  ##
  ## OPTIONAL INPUTS -- concentrated out if not supplied:
  ## K0P        : N*1       intercept in VAR for Z
  ## K1P        : N*N       mean reversion matrix in VAR for Z
  ## sigma.e    : scalar    standard error of yield observation errors
  ##
  ## OUTPUT:
  ## llk        : T*1       time series of -log likelihoods (includes 2-pi constants)
  ## A          : 1*J       yt = A' + B'*Zt  (yt is J*1 vector)
  ## B          : cN*J
  ## ...

########################################################################
  ## Setup
  T <- nrow(yields.o)-1
  J <- ncol(yields.o)
  cM <- ncol(M.o)
  cN <- cL + cM
  WL <- matrix(W[1:cL,], cL, J)
  PL.o <- ts( yields.o %*% t(WL))
  Z <- cbind(M.o, PL.o)
  Omega <- Sigma %*% t(Sigma)
########################################################################

########################################################################
  ## COMPUTE THE Q-LIKELIHOOD:
  ## First find the loadings for the model:
  ## yt = A' + B'*Zt, A is 1*J, B is cN*J

  loads <- jls.loadings(W, rinfQ, lamQ, gam0, gam1, Omega, mats, dt)
  B <- loads$B; A <- loads$A;

  yields.m <- rep(1,T+1)%*%A + Z %*% B # (T+1)*J, model-implied yields
  yield.errors <- yields.o[2:(T+1),] - yields.m[2:(T+1),]; # T*J
  square_orthogonal_yield.errors <- yield.errors^2; # T*J, but cL-dimensional projection onto W is always 0, so effectively (J-cL) dimensional

  ## Compute optimal sigma.e if it is not supplied
  if (is.na(sigma.e))
    sigma.e <- sqrt( sum(square_orthogonal_yield.errors)/(T*(J-cL)) )

  llkQ <- .5*rowSums(square_orthogonal_yield.errors)/sigma.e^2 + (J-cL)*.5*log(2*pi) + .5*(J-cL)*log(sigma.e^2) # 1*T
########################################################################

########################################################################
  ## COMPUTE THE P-LIKELIHOOD:

  if (missing(K0P)|missing(K1P)) {
    ## Run OLS to obtain maximum likelihood estimates of K0P, K1P
    var1 <- ar.ols(Z, order.max=1, aic=FALSE, demean=FALSE, intercept=TRUE)
    K1P <- var1$ar[,,] - diag(cN)
    K0P <- var1$x.intercept
  }

  innovations = t(Z[2:(T+1),]) - (K0P%*%matrix(1,1,T) + (K1P+diag(cN))%*%t(Z[1:T,])) # N*T

  llkP = .5*cN*log(2*pi) + .5*log(det(Omega)) + .5*colSums(innovations*solve(Omega, innovations)) # 1*T

########################################################################

  jsz.llk <- list(llk=t(llkQ + llkP), A=A, B=B, K0P=K0P, K1P=K1P, sigma.e=sigma.e, K0Q=loads$K0Q, K1Q=loads$K1Q, rho0=loads$rho0, rho1=loads$rho1, llkQ=llkQ, llkP=llkP)

}

jls.fkf <- function(Y, M.o, W, cL, rinfQ, lamQ, gam0, gam1, K0P, K1P, Sigma, mats, sigma.e, flag.filter) {
    ## Fast Kalman Filter for JLS model
    ## less general than jls.kalman: no macro measurement errors, no smoothing
    require(FKF)
    J <- length(mats)
    cM <- ncol(M.o)
    cN <- cL + cM
    Omega <- Sigma %*% t(Sigma)
    Phi <- K1P + diag(cN)
    mu <- K0P
    loads <- jls.loadings(W, rinfQ, lamQ, gam0, gam1, Omega, mats, dt=1)

    ## initial obs: unconditional moments
    a0 <- as.vector(solve(diag(cN)-Phi) %*% mu)
    P0 <- matrix( solve( diag(cN^2) - kronecker(Phi, Phi) ) %*%as.numeric(Omega), cN, cN);

    ## transition equation
    dt <- matrix(mu, cN, 1)
    Tt <- array(Phi, c(cN, cN, 1))
    HHt <- array(Omega, c(cN, cN, 1))

    ## measurement equation
    ct <- matrix(c(W %*% t(loads$A), rep(0, cM)), J+cM, 1)
    Zt <- array(rbind(W %*% t(loads$B), cbind(diag(cM), matrix(0, cM, cL))), c(J+cM, cN, 1))
    GGt.P <- W %*% diag(rep(sigma.e^2,J)) %*% t(W)
    if (!flag.filter) {
        GGt.P[1:cL, 1:cL] <- 0
    }
    GGt <- array(rbind(cbind(GGt.P, matrix(0,J,cM)),
                       matrix(0,cM,J+cM)),
                 c(J+cM, J+cM, 1))

    yt <- t(cbind(Y %*% t(W), M.o))

    rval <- fkf(a0, P0, dt, ct, Tt, Zt, HHt, GGt, yt)

    return(c(rval, list(A=loads$A, B=loads$B, K0Q=loads$K0Q, K1Q=loads$K1Q, rho0=loads$rho0, rho1=loads$rho1)))
}

jls.kalman <- function(yields.o, M.o, W, cL, rinfQ, lamQ, gam0, gam1, K0P, K1P, Sigma, mats, dt, sigma.e, sigma.m=NA, smoothing=FALSE) {
  ## setup
  T <- nrow(yields.o)
  J <- ncol(yields.o)
  cM <- ncol(M.o)
  cN <- cL + cM
  Omega <- Sigma %*% t(Sigma)
  Phi <- K1P + diag(cN)
  mu <- K0P

  ## parameters state equation
  Q <- Omega

  ## parameters measurement equation
  ## 1. J equations for yields         yt^o = A + B * Zt + e_t
  loads <- jls.loadings(W, rinfQ, lamQ, gam0, gam1, Omega, mats, dt)
  Omega.e <- diag(J)*sigma.e^2
  ## 2. cM equations for macro factors Mt^o = Mt + eta_t
  if (any(is.na(sigma.m))) {
    Omega.M <- matrix(0, cM, cM) # observed without error
  } else {
    if (length(sigma.m)!=cM) stop("length of sigma.m is wrong")
    Omega.M <- diag(sigma.m)^2
  }

  G <- c(as.numeric(loads$A), rep(0, cM))   # (J+cM) x 1
  H <- rbind(t(loads$B), cbind(diag(cM),matrix(0, cM, cL)))   # (J+cM) x cN
  R <- rbind( cbind(Omega.e, matrix(0,J,cM)), cbind(matrix(0,cM,J), Omega.M) ) # (J+cM) x (J+cM)

  T <- dim(yields.o)[1]
  x.tt <- matrix(NA, T, cN)
  x.ttm1 <- matrix(NA, T, cN)
  P.tt <- array(NA, c(T, cN, cN))
  P.ttm1 <- array(NA, c(T, cN, cN))

  ## starting values for x and P
  x.ttm1[1,] <- solve(diag(cN) - Phi) %*% mu
  P.ttm1[1,,] <- matrix( solve(diag(cN^2) - kronecker(Phi, Phi))%*%as.numeric(Q), cN, cN);

  llk <- numeric(T)
  for (t in 1:T) {
    V <- H %*% P.ttm1[t,,] %*% t(H) + R  # MSFE, cN x cN
    if (rcond(V)<1e-14) {
      return(list(llk=1000))
    }
    V.inv <- solve(V)
    K <- Phi %*% P.ttm1[t,,] %*% t(H) %*% V.inv  # Kalman gain
    eta <- c(yields.o[t,], M.o[t,]) - G - H %*% x.ttm1[t,]
    if (det(V)<=0) {
      return(list(llk=1000))
    }

    llk[t] <- -.5*log(det(V)) -.5 * t(eta) %*% V.inv %*% eta

    x.tt[t,] <- x.ttm1[t,] + P.ttm1[t,,] %*% t(H) %*% V.inv %*% eta
    P.tt[t,,] <-  P.ttm1[t,,] - P.ttm1[t,,] %*% t(H) %*% V.inv %*% H %*% P.ttm1[t,,] #P_t|t

    if (t<T) {
      ## update next periods values
      x.ttm1[t+1,] <- mu + Phi %*% x.ttm1[t,] + K %*% eta  # x_t+1|t
      ## should be equal to mu + Phi %*% x.tt[t]
      P.ttm1[t+1,,] <- Phi %*% P.tt[t,,] %*% t(Phi) + Q  # P_t+1|t
    }
  }

  if (smoothing) {
    ## smoothing
    x.tT <- matrix(NA, T, cN)
    x.tT[T,] <- x.tt[T,]
    for (t in (T-1):1) {
      J.t <- P.tt[t,,] %*% t(Phi) %*% solve(P.ttm1[t+1,,])
      x.tT[t,] <- x.tt[t,] + J.t %*% (x.tT[t+1,] - x.ttm1[t+1,])
    }
    return(list(llk=-llk, x.tt=x.tt, x.tT=x.tT, A=loads$A, B=loads$B, K0Q=loads$K0Q, K1Q=loads$K1Q, rho0=loads$rho0, rho1=loads$rho1))
  } else {
    return(list(llk=-llk, x.tt=x.tt, A=loads$A, B=loads$B, K0Q=loads$K0Q, K1Q=loads$K1Q, rho0=loads$rho0, rho1=loads$rho1))
  }
}

########################################################################
########################################################################

jls.loadings <- function(W, rinfQ, lamQ, gam0, gam1, Omega, mats, dt) {
  ## Inputs:
  ##   W          : J*J,      vector of portfolio weights
  ##   rinfQ      : scalar,   long run mean
  ##   lamQ       : N*1,      Q-eigenvalues
  ##   gam0       : cM*1      spanning parameters -- intercept
  ##   gam1       : cM*cN     spanning parameters -- coefficients
  ##   Omega      : cN*cN,    cov of innovations to Z
  ##   mats       : 1*J,      maturities in years
  ##   dt         : scalar,   length of period in years
  ##
  ## Returns:
  ##   A    : 1*J
  ##   B    : N*J
  ##   K0Q  : N*1
  ##   K1Q  : N*N
  ##   rho0 : scalar
  ##   rho1 : N*1

  J <- length(mats)
  cN <- length(lamQ)
  cM <- 2 ## default
  cL <- cN - cM
  WN <- matrix(W[1:cN,], cN, J)
  mats.periods <- round(mats/dt)
  M <- max(mats.periods)

  ## 1. based on primitive parameters, find (rho0, rho1, K0Q, K1Q)
  Gam1 <- rbind(gam1, cbind(diag(cL), matrix(0, cL, cM)))
  Gam0 <- rbind(gam0, matrix(0, cL,1))

  ## 1.1. find loadings of yields on Xt (Jordan-normalized factors)
  K0Q.X <- matrix(0, cN, 1);
  K1Q.X <- diag(lamQ)
#  K1Q.X <- jszAdjustK1QX(K1Q.X)$K1Q.X
  rho0.X <- rinfQ;   rho1.X <- rep(1, cN)
  ## 1.1.1. first compute the loadings ignoring convexity term
  loads.X.prelim <- gaussian.loadings(mats.periods, K0Q.X, K1Q.X, matrix(0, cN, cN), rho0.X*dt, rho1.X*dt, dt)
  BX <- loads.X.prelim$B  ## cN * J
  U1inv <- Gam1 %*% WN %*% t(BX)
  U1 <- solve(U1inv)
  ## 1.1.2. calc. Omega.X and calculate correct loadings
  Omega.X <- U1 %*% Omega %*% t(U1)
  loads.X <- gaussian.loadings(mats.periods, K0Q.X, K1Q.X, Omega.X, rho0.X*dt, rho1.X*dt, dt)
  AX <- loads.X$A ## 1 * J

  ## 1.2. calculate remaining parameters
  U0 <- Gam0 + Gam1 %*% WN %*% t(AX)
  rho1 <- t(U1) %*% rep(1, cN)
  rho0 <- rinfQ - crossprod(rho1, U0)
  K1Q <- U1inv %*% K1Q.X %*% U1
  K0Q <- -K1Q %*% U0

  ## 2. compute affine loadings for Z
  loads.Z <- gaussian.loadings(mats.periods, K0Q, K1Q, Omega, rho0*dt, rho1*dt, dt)

############################################################
  jls.loadings <- list(A=loads.Z$A, B=loads.Z$B, K0Q=K0Q, K1Q=K1Q, rho0=rho0, rho1=rho1, U0=U0, U1=U1, Gam0=Gam0, Gam1=Gam1, AX=AX, BX=BX)

}

########################################################################
########################################################################

gaussian.loadings <- function(maturities, K0d, K1d, H0d, rho0d, rho1d, timestep=1) {
  ## maturities: M*1
  ## K0d      : N*1
  ## K1d      : N*1
  ## H0d      : N*N
  ## rho0d    : scalar
  ## rho1d    : N*1
  ## timestep : optional argument.
  ##
  ## By : N*M
  ## Ay : 1*M  (faster to not compute with only one output argument)
  ##
  ## r(t)   = rho0d + rho1d'Xt
  ##        = 1 period discount rate
  ## P(t)   =  price of  t-period zero coupon bond
  ##        = EQ0[exp(-r0 - r1 - ... - r(t-1)]
  ##        = exp(A+B'X0)
  ## yields = Ay + By'*X0
  ##   yield is express on a per period basis unless timestep is provided.
  ##   --For example, if the price of a two-year zero is exp(-2*.06)=exp(-24*.005),
  ##   --and we have a monthly model, the function will return Ay+By*X0=.005
  ##   --unless timestep=1/12 is provided in which case it returns Ay+By*X0=.06
  ##
  ## Where under Q:
  ##   X(t+1) - X(t) = K0d + K1d*X(t) + eps(t+1),  cov(eps(t+1)) = H0d
  ##
  ## A1 = -rho0d
  ## B1 = -rho1d
  ## At = A(t-1) + K0d'*B(t-1) .5*B(t-1)'*H0d*B(t-1) - rho0d
  ## Bt = B(t-1) + K1d'*B(t-1) - rho1d
  ##
  ## maturities: 1*M # of periods

  M = length(maturities)
  N = length(K0d)
  Atemp = 0
  Btemp = matrix(0,N,1)
  Ay = matrix(NA,1,M)
  By = matrix(NA,N,M)

  curr_mat = 1
  K0dp <- t(K0d)
  K1dp <- t(K1d)
  for (i in 1:maturities[M]) {
    Atemp <- Atemp + K0dp%*%Btemp +.5%*%t(Btemp)%*%H0d%*%Btemp - rho0d
    Btemp <- Btemp + K1dp%*%Btemp - rho1d

    if (i==maturities[curr_mat]) {
      Ay[1,curr_mat] <- -Atemp/maturities[curr_mat]
      By[,curr_mat] <- -Btemp/maturities[curr_mat]
      curr_mat <- curr_mat + 1
    }
  }

  gaussian.loadings <- list(A = Ay/timestep, B = By/timestep)
}

########################################################################

jszAdjustK1QX <- function(K1Q.X, eps1=1e-3) {
  ## function [K1Q_X, isTypicalDiagonal, m1] = jszAdjustK1QX(K1Q_X, eps1);
  ##
  ## This function adjusts diagonal K1Q_X to give a non-diagonal but more
  ## computationally tractable K1Q_X.
  ##
  ## K1Q_X can fall into a few cases:
  ##   0. diagonal
  ##   1. not diagonal
  ##   2. zero eigenvalue
  ##   3. near repeated roots
  ## In cases 1-3, the diagonal closed form solver doesn't work, so compute differently.
  ## In case 1-2, we will use the recursive solver, though there are more efficient methods.
  ## In case 3, we will add a positive number above the diagonal.  this gives a different member of the set of observationally equivalent models.
  ##   So for example:
  ##      [lambda1, 0; 0, lambda2] is replaced by [lambda1, f(lambda2-lambda1); 0, lambda2] when abs(lambda1 - lambda2)<eps0
  ##   By making f not just 0/1, it will help by making the likelihood
  ##   continuous if we parameterize by kinf. (not an issue in some cases.)
  ##
  ## We also order the diagonal of diagonal K1Q.


  ## Cutoff function sets the super diagonal entry to something between 0 and
  ## 1, depending on how close the eigenvalues are.
  cutoff.fun <- function(x, eps1) {
    eps1 = 1e-3;
    eps0 = 1e-5;
    ##    xc <- 1*(x<eps0) + (1 - (x - eps0)/(eps1 - eps0))*(x>=eps0 && x<eps1) + 0*(x>eps1);
    xc <- 1*(log(x)<log(eps0)) +
      (1 - (log(x) - log(eps0))/(log(eps1) - log(eps0)))*(log(x)>=log(eps0) & log(x)<log(eps1)) +
        0*(log(x)>log(eps1));
    xc[x==0] <- 1;
    return(xc)
  }

  N <- nrow(K1Q.X)

  diag.K1Q.X <- diag(K1Q.X);
  isDiagonal <- all(K1Q.X==diag(diag.K1Q.X));

  ## For diagonal matrix, sort the diagonal and check to see if we have near repeated roots.
  if (isDiagonal) {
    diag.K1Q.X <- -sort(-diag.K1Q.X);
    K1Q.X <- diag(diag.K1Q.X);

    hasNearUnitRoot <- !all(abs(diag.K1Q.X)>eps1); ## Only applicable for diagonal
    hasNearRepeatedRoot <- !all(abs(diff(diag.K1Q.X))>eps1); ## Only applicable for diagonal
    isTypicalDiagonal <- isDiagonal && !hasNearRepeatedRoot && !hasNearUnitRoot;
  } else {
    isTypicalDiagonal <- FALSE
  }

  ## If we have a near repeated root, add a constnat above the diagonal. This
  ## representative of the equivalence class gives easier inversion for latent
  ## states vs. yields.  By varying the constant

  if (isDiagonal && !isTypicalDiagonal) {
    diff.diag <- abs(diff(diag.K1Q.X));
    super.diag <- cutoff.fun(diff.diag);
    K1Q.X[1:N-1,2:N] <- K1Q.X[1:N-1,2:N] + diag(super.diag);
  }
#######################################

  super.diag = diag(K1Q.X[-N,-1]);
  m1 <- max(which(cumprod(c(1,super.diag))>0))

  return(list(K1Q.X=K1Q.X, isTypicalDiagonal=isTypicalDiagonal, m1=m1))

}
