function [param, ss] = calib
% Calibrate parameters and compute steady-state equilibrium in Chang, Liu, and Spiegel (2015 JME)

%---Preference parameters
beta = 0.995;   % subjective discount factor
chi = 0;        % Habit persistence parameter (NOT USED)
eta = 2;        % Inverse Frisch elasticity of labor supply 
Phi_m = 0.06;   % utility weight on real money balances 

%--Technology parameters ---
phi = 0.5;       % cost share of intermediate goods in production 
lambda_z = 1.02; %annual GDP per capita growth rate 


%--Parameters for nominal ridigities ----
theta_p = 10;   %elasticity of substitution between differentiated products
Omega_p = 60;   % price adjustment cost parameter 

%---Parameters for home bias and portfolio adjustments ---
psi_bar = 0.9;  % steady-state portfolio share of domestic bonds in private sector (home bias)
Omega_b = 0.6; % Portfolio adjustment cost parameter

%--Parameters for international trade ----
theta = 1.5;    % export demand elasticity

% Steady-state targets and model-implied values
ge=1;                                   % Growth rate of nominal exchange rate
q=1;                                    % Real exchange rate
R_star = 1.0175;                        % Foreign nominal interest rate 
pi_star = 1.005;                        % Foreing inflation rate 
Im_GDP_ratio = 0.20;                    % Averag import-to-GDP ratio =0.20 from 1990 to 2009
Ex_GDP_ratio = 0.23;                    % Average export-to-GDP ratio = 0.23 from 1990 to 2009
L = 0.4;                                % targeted labor supply as a fraction of time endowment
Z_m = 1;                                % mean level of transitory TFP shock (NOT USED)
R = R_star*ge;                          % Nominal interest rate from UIP
pi = pi_star*R/R_star;                  % Inflation rate
mu = pi*lambda_z;                       % Money growth rate 
v = (theta_p-1)/theta_p;                % Real marginal cost
alpha = 1- (1/(phi*v)-1)*Im_GDP_ratio;  % Share of imported goods in intermediate input
alpha_tilde = alpha^(-alpha)*(1-alpha)^(alpha-1);  
phi_tilde = phi^(-phi)*(1-phi)^(phi-1); 
q_m = alpha_tilde*q^(1-alpha);          % Relative price of intermediate goods
Y = (phi*v/q_m)^(phi/(1-phi))*Z_m*L;    % Gross output
GDP = (1-phi*v)*Y;                      % Real GDP
NX = GDP*(Ex_GDP_ratio-Im_GDP_ratio);   % Net exports 
X = Ex_GDP_ratio*GDP;                   % Exports
G = 0;                                  % Government spending
C = GDP-NX-G;                           % Consumption 
X_star = X/q^theta;                     % Implied foreign demand for export (targeting q=1)
Lambda = 1/C;                           % Marginal utility
w = (1-phi)*v*Y/L;                      % Real wage
Phi_l = w*Lambda/L^eta;                 % utility weight on labor hours
Gam_f = Im_GDP_ratio*GDP/q;             % Imported intermediate goods
Gam_h = alpha*phi*v*Y;                  % Domestically produced intermediate goods
Gam = phi*v*Y/q_m;                      % Total intermediate input
m = (Phi_m/Lambda)*R/(R-1);             % Real money balances
b = (NX-m*(1-1/mu))*psi_bar/(1-R/(pi*lambda_z));  % Domestic bond 
bstar_p = (1-psi_bar)*b/(q*psi_bar);     % Private holdings of foreign bond
bstar = NX/(q*(1-R_star/(pi_star*lambda_z)));   % Total foreign bond
bstar_g = bstar-bstar_p;                % Central bank holdings of foreign bond
ca = q*bstar*(1-1/(pi_star*lambda_z));  % Current account balance


param = [alpha, chi, beta, eta, theta_p, Omega_p, Omega_b, phi, Phi_m, Phi_l, psi_bar, theta]';
ss = [C, L, w, R, pi, Lambda, m, q_m, v, Gam, Gam_h, Gam_f, q, ca, X, bstar, b, Y, GDP, mu, lambda_z, R_star, pi_star, X_star, ge,G, bstar_p, bstar_g]';



